/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* TABLE.C -- Scrollable and key-selectable tables of strings. */
#include <stdlib.h>
#include <assert.h>
#include <graphics.h>
#include <string.h>
#include "window.h"
#include "resource.h"
#include "key.h"

enum {

table_x_margin = 4,
table_y_margin = 2,
table_border_width = 1,
table_border_color = cBLACK,
table_bg_color = cWHITE,
table_text_color = cBLACK,
table_disabled_text_color = cLIGHTGRAY,
table_cursor_color = cLIGHTGRAY,
table_hi_cursor_color = cLIGHTRED,
table_button = bLEFT,

};

static char *pbuf;
static int pbuf_len, pp;

LOCAL(void) draw_entry(int pos, TABLE tbl)
{
  int entry_width, x0, x1, y0, y1;
  div_t r;
#define rrow r.rem
#define rcol r.quot

  r = div(pos, tbl->wrap);
  rrow -= tbl->row;
  rcol -= tbl->col;
  if (rrow < 0 || rrow >= tbl->n_vis_rows ||
      rcol < 0 || rcol >= tbl->n_vis_cols)
    return;

  entry_width = table_x_margin + tbl->entry_len * tbl->ch_width + table_x_margin;
  x0 = rcol * (entry_width + table_border_width);
  y0 = table_y_margin + rrow * tbl->ch_height;
  y1 = y0 + tbl->ch_height - 1;
  push_graphics_state(&tbl->window, 0);
  settextjustify(LEFT_TEXT, TOP_TEXT);
  setcolor(table_text_color);
  protect_cursor(&tbl->window);
  if (pos == tbl->pos) {
    if (t_status_p(tbl, tHAVE_FOCUS)) {
      x1 = x0 + table_x_margin + pp * tbl->ch_width;
      setfillstyle(SOLID_FILL, table_hi_cursor_color);
      bar(x0, y0, x1, y1);
    }
    else 
      x1 = x0;
    setfillstyle(SOLID_FILL, table_cursor_color);
  }
  else {
    x1 = x0;
    setfillstyle(SOLID_FILL, table_bg_color);
  }
  bar(x1, y0, x0 + entry_width - 1, y1);
  outtextxy(x0 + table_x_margin, y0, tbl->entries[pos]);
  unprotect_cursor();
  pop_graphics_state();
}
#undef rrow
#undef rcol

LOCAL(void) draw_table(TABLE tbl)
{
  int entry_width, vis_col, col_pos, vis_row, pos, tbl_y1, x0, x1, y0, y1;

  if (!w_status_p(&tbl->window, wsVISIBLE))
    return;

  entry_width = table_x_margin + tbl->entry_len * tbl->ch_width + table_x_margin;
  tbl_y1 = 2*table_y_margin + tbl->ch_height * tbl->n_vis_rows - 1;

  push_graphics_state(&tbl->window, 0);
  protect_cursor(&tbl->window);
  setcolor(t_status_p(tbl, tENABLED) ? table_text_color : table_disabled_text_color);
  settextjustify(LEFT_TEXT, BOTTOM_TEXT);
  out_hot_textxy(0, -table_y_margin - 1, &tbl->title);

  settextjustify(LEFT_TEXT, TOP_TEXT);
  for (vis_col = 0, col_pos = tbl->col * tbl->wrap + tbl->row, x0 = 0;
       vis_col < tbl->n_vis_cols;
       ++vis_col, col_pos += tbl->wrap, x0 += (entry_width + table_border_width)) {

    if (vis_col > 0) {
      setfillstyle(SOLID_FILL, table_text_color);
      bar(x0 - table_border_width, 0, x0 - 1, tbl_y1);
    }

    for (vis_row = 0, pos = col_pos, y0 = table_y_margin, y1 = y0 + tbl->ch_height - 1;
	 vis_row < tbl->n_vis_rows;
	 ++vis_row, ++pos, y0 += tbl->ch_height, y1 += tbl->ch_height) {
      if (pos == tbl->pos) {
	if (t_status_p(tbl, tHAVE_FOCUS)) {
	  x1 = x0 + table_x_margin + pp * tbl->ch_width;
	  setfillstyle(SOLID_FILL, table_hi_cursor_color);
	  bar(x0, y0, x1, y1);
	}
	else
	  x1 = x0;
	setfillstyle(SOLID_FILL, table_cursor_color);
      }
      else {
	x1 = x0;
	setfillstyle(SOLID_FILL, table_bg_color);
      }
      bar(x1, y0, x0 + entry_width - 1, y1);
      if (pos < tbl->n_entries)
	outtextxy(x0 + table_x_margin, y0, tbl->entries[pos]);
    }
  }

  unprotect_cursor();
  pop_graphics_state();

  if (t_status_p(tbl, tENABLED)) {
    tbl->window.event_mask |= bit(eBUTTON_PRESS);
    if (tbl->hor_scrollbar != NULL)
      enable_scrollbar(tbl->hor_scrollbar);
    if (tbl->ver_scrollbar != NULL)
      enable_scrollbar(tbl->ver_scrollbar);
  }
  else {
    tbl->window.event_mask &= notbit(eBUTTON_PRESS);
    if (tbl->hor_scrollbar != NULL)
      disable_scrollbar(tbl->hor_scrollbar);
    if (tbl->ver_scrollbar != NULL)
      disable_scrollbar(tbl->ver_scrollbar);
  }
}

void locate_table_cursor(int pos, TABLE tbl)
{
  int dx, dy;
  div_t r;
#define rrow r.rem
#define rcol r.quot

  if (pos < 0 || pos >= tbl->n_entries) {
    tbl->pos = -1;
    draw_table(tbl);
    return;
  }

  r = div(pos, tbl->wrap);
  dx = dy = 0;
  if (rrow < tbl->row)
    dy = rrow - tbl->row;
  else {
    int last_row = tbl->row + tbl->n_vis_rows - 1;
    if (rrow > last_row)
      dy = rrow - last_row;
  }
  if (rcol < tbl->col)
    dx = rcol - tbl->col;
  else {
    int last_col = tbl->col + tbl->n_vis_cols - 1;
    if (rcol > last_col)
      dx = rcol - last_col;
  }
  if (dx || dy) {
    tbl->row += dy;
    if (tbl->ver_scrollbar) 
      position_scrollbar(tbl->ver_scrollbar, tbl->row);
    tbl->col += dx;
    if (tbl->hor_scrollbar) 
      position_scrollbar(tbl->hor_scrollbar, tbl->col);
    tbl->pos = pos;
    draw_table(tbl);
  }
  else {
    int old_pos = tbl->pos;
    tbl->pos = pos;
    draw_entry(old_pos, tbl);
    draw_entry(pos, tbl);
  }
}

#undef rrow
#undef rcol

LOCAL(void) handle_map(EVENT e)
{
  draw_table((TABLE)e->map.window);
}

static void handle_ver_scroll(int pos, ENV env)
#define tbl ((TABLE) env)
{
  tbl->row = pos;
  draw_table(tbl);
}
#undef tbl

static void handle_hor_scroll(int pos, ENV env)
#define tbl ((TABLE) env)
{
  tbl->col = pos;
  draw_table(tbl);
}
#undef tbl

LOCAL(void) handle_button_press(EVENT e)
{
  int row, col, pos;
  TABLE tbl;

  if (e->mouse.button != table_button)
    return;

  tbl = (TABLE)e->mouse.window;
  pp = 0;
  set_focus(e->mouse.window);
  row = max(0, min(tbl->n_vis_rows - 1, (e->mouse.y - table_y_margin) / tbl->ch_height));
  col = max(0, min(tbl->n_vis_cols - 1, e->mouse.x / (table_x_margin + tbl->entry_len * tbl->ch_width + table_x_margin + table_border_width)));
  row += tbl->row;
  col += tbl->col;
  pos = col * tbl->wrap + row;
  if (pos < tbl->n_entries)
    if (double_click_p(&tbl->entries[pos])) {
      if (tbl->n_key_actions > 0)
	(*tbl->key_actions[0].action.code)(tbl->entries[pos], tbl->key_actions[0].action.env);
    }
    else
      locate_table_cursor(pos, tbl);
}

void set_table_entries(TABLE tbl, char **entries, int n_entries)
{
  SCROLLBAR sb;

  pp = 0;
  tbl->row = tbl->col = 0;
  tbl->entries = entries;
  if (entries == NULL) {
    tbl->pos = -1;
    tbl->n_entries = 0;
  }
  else {
    tbl->pos = t_status_p(tbl, tENABLED) ? 0 : -1;
    tbl->n_entries = n_entries;
  }
  sb = tbl->ver_scrollbar;
  if (sb != NULL) {
    sb->n_pos = max(1, min(tbl->n_entries, tbl->wrap) - tbl->n_vis_rows + 1);
    sb->n_step_pos = tbl->n_vis_rows;
    position_scrollbar(sb, 0);
  }

  sb = tbl->hor_scrollbar;
  if (sb != NULL) {
    div_t r = div(tbl->n_entries, tbl->wrap);
    sb->n_pos = max(1, r.quot + (r.rem > 0) - tbl->n_vis_cols + 1);
    sb->n_step_pos = tbl->n_vis_cols;
    position_scrollbar(sb, 0);
  }
  draw_table(tbl);
}

LOCAL(void) handle_gain_focus(EVENT e)
{
  TABLE tbl = (TABLE)e->focus.window;
  pbuf = malloc(pbuf_len = tbl->entry_len);
  pp = 0;
  tbl->status |= bit(tHAVE_FOCUS);
  draw_table(tbl);
}

LOCAL(void) handle_lose_focus(EVENT e)
{
  TABLE tbl = (TABLE)e->focus.window;
  free(pbuf);
  tbl->status &= notbit(tHAVE_FOCUS);
  draw_table(tbl);
}

int table_match_pos(char *pattern, int plen, char **entries, int n_entries)
{
  char **ep, **end;

  for (ep = entries, end = ep + n_entries; ep < end; ++ep)
    if (strnicmp(pattern, *ep, plen) == 0)
      return (ep - entries);
  return -1;
}

LOCAL(void) handle_keystroke(EVENT e)
{
  int pos, key;
  TABLE tbl;
  KEY_NAME_ACTION p, end;

  key = e->keystroke.key;
  tbl = (TABLE)e->keystroke.window;
  for (p = tbl->key_actions, end = p + tbl->n_key_actions; p < end; ++p)
    if (p->key == key) {
      (*p->action.code)(tbl->pos == -1 ? NULL : tbl->entries[tbl->pos], p->action.env);
      return;
    }

  switch (key) {

    case UP_ARROW:
      if (tbl->pos > 0) {
	pp = 0;
	locate_table_cursor(tbl->pos - 1, tbl);
      }
      break;

    case DOWN_ARROW:
      if (tbl->pos < tbl->n_entries - 1) {
	pp = 0;
	locate_table_cursor(tbl->pos + 1, tbl);
      }
      break;

    case LEFT_ARROW:
      if (tbl->wrap != NO_WRAP && tbl->pos > 0) {
	pp = 0;
	locate_table_cursor(max(0, tbl->pos - tbl->wrap), tbl);
      }
      break;

    case RIGHT_ARROW:
      if (tbl->wrap != NO_WRAP && tbl->pos < tbl->n_entries - 1) {
	pp = 0;
	locate_table_cursor(min(tbl->n_entries - 1, tbl->pos + tbl->wrap), tbl);
      }
      break;

    case HOME:
      pp = 0;
      locate_table_cursor(0, tbl);
      break;

    case END:
      pp = 0;
      locate_table_cursor(tbl->n_entries - 1, tbl);
      break;

    case '\b':
      if (pp > 0)
	--pp;
      goto locate_cursor_at_match;

    default:
      if (!ctrl_keystroke_p(e) && pp < pbuf_len) {
	pbuf[pp++] = e->keystroke.key;

      locate_cursor_at_match:
	pos = table_match_pos(pbuf, pp, tbl->entries, tbl->n_entries);
	if (pos == -1) {
	  pp = 0;
	  locate_table_cursor(tbl->pos, tbl);
	}
	else
	  locate_table_cursor(pos, tbl);
      }
      else
	refuse_keystroke(e);
      break;
  }
}

void enable_table(TABLE tbl)
{
  if (t_status_p(tbl, tENABLED))
    return;

  tbl->window.event_mask |= bit(eBUTTON_PRESS)|bit(eGAIN_FOCUS)|bit(eKEYSTROKE)|bit(eVISIBLE_HOTKEY);
  tbl->status |= bit(tENABLED);
  tbl->pos = 0;
  draw_table(tbl);
}

void disable_table(TABLE tbl)
{
  unset_focus(&tbl->window);
  tbl->window.event_mask &= notbit(eBUTTON_PRESS)&notbit(eGAIN_FOCUS)&notbit(eKEYSTROKE)&notbit(eVISIBLE_HOTKEY);
  tbl->status &= notbit(tENABLED);
  tbl->pos = -1;
  draw_table(tbl);
}

LOCAL(void) handle_hotkey(EVENT e)
{
  TABLE tbl = (TABLE)e->hotkey.window;
  if (key_hot_p(e->hotkey.key, &tbl->title))
    set_focus(&tbl->window);
}

BeginDefDispatch(table)
  Dispatch(eMAP, handle_map)
  Dispatch(eBUTTON_PRESS, handle_button_press)
  Dispatch(eKEYSTROKE, handle_keystroke)
  Dispatch(eGAIN_FOCUS, handle_gain_focus)
  Dispatch(eLOSE_FOCUS, handle_lose_focus)
  Dispatch(eVISIBLE_HOTKEY, handle_hotkey)
EndDefDispatch(table)

void open_table(TABLE tbl, int x, int y, WINDOW parent)
{
  int width, height, hor_scrollbar_p, ver_scrollbar_p;
  unsigned mask;

  tbl->ch_width = textwidth("W");
  tbl->ch_height = textheight("Hy");

  /* Decide what scrollbars to use. */
  hor_scrollbar_p = tbl->wrap != NO_WRAP;
  ver_scrollbar_p = tbl->n_vis_rows < tbl->wrap;

  /* Store into width, height the size of the table window. */
  width = (table_x_margin +
	   tbl->entry_len * tbl->ch_width +
	   table_x_margin +
	   table_border_width) * tbl->n_vis_cols -
	   table_border_width;
  if (ver_scrollbar_p)
    width += scrollbar_thickness(1);

  height = table_y_margin +
	   tbl->n_vis_rows * tbl->ch_height +
	   table_y_margin;
  if (hor_scrollbar_p)
    height += scrollbar_thickness(1);

  /* Build the table window. */
  mask = bit(eMAP)|bit(eLOSE_FOCUS);
  if (t_status_p(tbl, tENABLED))
    mask |= bit(eBUTTON_PRESS)|bit(eGAIN_FOCUS)|bit(eKEYSTROKE)|bit(eVISIBLE_HOTKEY);
  open_window(&tbl->window, parent,
	      x, y, width, height,
	      table_border_width, table_border_color, table_bg_color,
	      mask);
  SetDispatch(&tbl->window, table);
  /* Build scrollbars. */
  if (hor_scrollbar_p) {
    tbl->hor_scrollbar = NULL;
    walloc(&tbl->hor_scrollbar, sizeof(SCROLLBAR_REC));
    SetScrollbar(tbl->hor_scrollbar, NoTitle, NoHotChar,
		 FullLength, 100, 1, 10, 0, 
		 Enabled|(ver_scrollbar_p ? OneOfTwo : 0), 
		 handle_hor_scroll, tbl);
    open_scrollbar(tbl->hor_scrollbar, DEFAULT, DEFAULT, &tbl->window);
    map_window(&tbl->hor_scrollbar->window);
  }
  else 
    tbl->hor_scrollbar = NULL;

  if (ver_scrollbar_p) {
    tbl->ver_scrollbar = NULL;
    walloc(&tbl->ver_scrollbar, sizeof(SCROLLBAR_REC));
    SetScrollbar(tbl->ver_scrollbar, NoTitle, NoHotChar,
		 FullLength, 100, 1, 10, 0, 
		 Enabled|Vertical|(hor_scrollbar_p ? OneOfTwo : 0), 
		 handle_ver_scroll, tbl);
    open_scrollbar(tbl->ver_scrollbar, DEFAULT, DEFAULT, &tbl->window);
    map_window(&tbl->ver_scrollbar->window);
  }
  else 
    tbl->ver_scrollbar = NULL;

  set_table_entries(tbl, tbl->entries, tbl->n_entries);
}
