/* Original Author:         Andrew Trevorrow
   Implementation: Modula-2 under VAX/UNIX 4.2 BSD
   Date Started:   June, 1986

   Description:
   Implements the InitVIS500()procedure that initializes the terminal dependent
   routines and parameters used in DVItoVDU.
   The VISUAL 500 has an Alphanumeric mode in which it behaves like a VT52.
   DVItoVDU assumes text lines on the screen start at 1 and increase
   downwards; MoveAbs()does the necessary translation to VT52 coordinates.
   Note that the VIS500 is very similar to a VT640 (except that the VT640 acts
   like a VT100 in what it calls Transparent mode).

   The VIS500 can also emulate a Tektronix 4010 terminal in which the screen
   is 780 pixels high by 1024 pixels wide.
   (The actual resolution of the VIS500 screen is 585 by 768 pixels;
   the appropriate 3:4 scaling is done automatically.)
   The bottom left pixel is the point (x=0,y=0). Note that x coordinates
   increase to the right and y coordinates increase up the screen.
   DVItoVDU uses a coordinate scheme in which horizontal (=h) coordinates
   also increase to the right but vertical (=v) coordinates increase DOWN the
   screen, i.e. the top left pixel on the screen is the point (h=0,v=0).
   This means that the Tektronix 4010 routines will have to do a
   simple translation of the vertical coordinates passed by DVItoVDU.

   For DVItoVDU to be able to display both Alphanumeric text and Tektronix
   mode graphics, the F6 function key must = BOTH.

   This version converted to C and ported to BSD and System V UNIX by
   some chaps at Kernel Technology up to September 1989.  Contact
   mjh@uk.co.kernel (Mark J. Hewitt) with bug fixes etc.

   Involved were:	Mark J. Hewitt
   			Dave Dixon
			Marc Hadley
*/

#include "def.h"

static char *sccsid[] = "@(#)vis500vdu.c	1.1";

extern void SendXYtek();

extern Void TEK4010ClearScreen(),
            TEK4010StartGraphics(),
            TEK4010StartText(),
            TEK4010ShowChar(),
            TEK4010LoadFont();

extern unsigned int dragdown;
extern short havesentxy;


Void VIS500MoveToTextLine ();
Void VIS500ClearTextLine ();
Void VIS500ClearScreen ();
Void VIS500ShowChar ();
Void VIS500ShowRectangle ();
Void VIS500ResetVDU ();
Void VIS500MoveAbs ();

/******************************************************************************/

void InitVIS500 ()
{

/* The dialog region will be the top 4 text lines in Alphanumeric mode:
      Line 1 = DVI status line,
      Line 2 = window status line,
      Line 3 = message line,
      Line 4 = command line.
   The window region will be text lines 5 to 33 in Alphanumeric mode.
*/

  DVIstatusl = 1;		/* DVItoVDU assumes top text line = 1 */
  windowstatusl = 2;
  messagel = 3;
  commandl = 4;
  bottoml = 33;			/* also number of text lines on VIS500 screen */
/* The above values assume the VIS500 is in Alphanumeric mode;
   the following values assume it is emulating a Tektronix 4010.
   Note that windowv must be given a value using DVItoVDU's coordinate scheme
   where top left pixel is (0,0).
*/
  windowv = 92;			/* approx. height in TEK4010 pixels of 4 text lines;
				   i.e. 4 * 780/34 */
  windowh = 0;
  windowht = 780 - windowv;
  windowwd = 1024;

  MoveToTextLine = VIS500MoveToTextLine;
  ClearTextLine = VIS500ClearTextLine;
  ClearScreen = VIS500ClearScreen;
  StartText = TEK4010StartText;
  StartGraphics = TEK4010StartGraphics;
  LoadFont = TEK4010LoadFont;
  ShowChar = VIS500ShowChar;
  ShowRectangle = VIS500ShowRectangle;
  ResetVDU = VIS500ResetVDU;

  Write (GS);
  Write (ESC);
  Write ('@');			/* solid fill for rectangular draw and fill */
  Write (CAN);
}

/******************************************************************************/

Void VIS500MoveToTextLine (line)
unsigned int  line;
{
  Write (CAN);			/* switch to Alphanumeric mode */
  VIS500MoveAbs (line, 1);
}

/******************************************************************************/

Void VIS500MoveAbs (row, col)
unsigned int  row, col;
{

/* Assuming we are in VT52 mode (Alphanumeric mode), move the cursor to the
   given row and column.  Since DVItoVDU assumes top left corner of screen
   is (row=1,col=1) we need to subtract 1 from both coordinates to get
   the actual VT52 position on the screen.
*/

  Write (ESC);
  Write ('Y');
  Write ((char) (((int) ' ') + row - 1));
  Write ((char) (((int) ' ') + col - 1));
}

/******************************************************************************/

Void VIS500ClearTextLine (line)
unsigned int  line;
{
  Write (CAN);			/* switch to Alphanumeric mode */
  VIS500MoveAbs (line, 1);	/* move to start of line */
  Write (ESC);
  Write ('K');			/* erase to end of line */
}

/******************************************************************************/

Void VIS500ClearScreen ()
{
  Write (CAN);			/* enable Alphanumeric display */
  Write (ESC);
  Write ('H');			/* move to top left of screen */
  Write (ESC);
  Write ('J');			/* erase to end of screen (Alphanumeric text) */
  TEK4010ClearScreen ();	/* erase graphics */
}

/******************************************************************************/

Void VIS500ShowChar (screenh, screenv, ch)/* ref point */
unsigned int  screenh, screenv;
char  ch;
/* The TEK4010 Terse mode characters on the VIS500 need to be dragged down
   so that any descenders will be below the baseline represented by screenv.
*/
{
  screenv = screenv + dragdown;
  if (screenv > 779)
  {
    TEK4010ShowChar (screenh, 779, ch);/* drag down as far as possible */
  }
  else
  {
    TEK4010ShowChar (screenh, screenv, ch);
  }
}

/******************************************************************************/

/*ARGSUSED*/
Void VIS500ShowRectangle (screenh, screenv,/* top left pixel */
  width, height,		/* of rectangle */
  ch)				/* black pixel */
unsigned int  screenh, screenv, width, height;
char  ch;
/* Display the given rectangle without using the given black pixel character.
   We assume that the top left position is correct and that the given
   dimensions do not go beyond the screen edges.
   If height and width > 1, we use the rectangular draw and fill command.
*/
{
  unsigned int  pos;

  if (height == 1)
  {				/* show row vector */
    pos = 779 - screenv;
    Write (GS);
    SendXYtek (screenh, pos);	/* move cursor to start of row */
    SendXYtek (screenh + width - 1, pos);/* draw vector to end of row */
  }
  else
    if (width == 1)
    {				/* show column vector */
      pos = 779 - screenv;
      Write (GS);
      SendXYtek (screenh, pos);	/* move cursor to start of column */
      SendXYtek (screenh, pos - height + 1);/* draw vector to end of column */
    }
    else
    {				/* assume height and width > 1; draw and fill
				   rectangle */
      pos = 779 - (screenv + height - 1);
      Write (ESC);
      Write ('/');
      WriteCard (screenh);
      Write (';');		/* left */
      WriteCard (pos);
      Write (';');		/* bottom */
      WriteCard (width - 1);
      Write (';');
      WriteCard (height + 1);
      Write ('y');
    /* Note that there are a few problems with this command: - we need to subtract 1
       from width.  While this prevents exceeding the right edge (reason unknown),
       it causes missing pixel columns. - we need to ADD 1 to height to avoid
       missing pixel rows. - the smallest rectangle drawn is 2 by 2. - the new
       cursor position is undefined. These funnies are outweighed by the improved
       efficiency in drawing large rectangles. */
      havesentxy = FALSE;	/* need to re-synch cursor position */
    }
}

/******************************************************************************/

Void VIS500ResetVDU ()
{
  Write (CAN);			/* make sure terminal is in Alphanumeric mode */
}

/******************************************************************************/
