
IMPLEMENTATION MODULE ScreenIO;

(* Author:         Andrew Trevorrow
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   September, 1984

   Description:
   Terminal i/o routines used by DVItoVDU.

   Revised:
   December, 1984
 - Users don't need to SET TERM /HOSTSYNC/TTSYNC/NOWRAP any more.
 - Implemented BusyRead and RestoreTerminal.

   May, 1985
 - Implemented Read, ReadString and Halt.

   August, 1985
 - Made BusyRead a bit more efficient.
 - For efficiency reasons, WriteCard and WriteInt no longer
   have a field width parameter.  (It was never used anyway!)

   September, 1986
 - No need to set terminal to /hostsync.

   November, 1988 (Niel Kempson <rmcs-tex@uk.ac.cranfield.cdvc>,
		  RMCS Shrivenham, SWINDON SN6 8LA, United Kingdom)
 - Updated to use VMS version 4.x system service calls,
   i.e. `channel' changed from type CARDINAL to SHORTWORD.
*)

  FROM SYSTEM IMPORT 
                     ADR, SHORTWORD;

  FROM VMS IMPORT 
                  SYS$ASSIGN, SYS$QIOW, SYS$EXIT;

  FROM IODefinitions IMPORT 
                            IO$_WRITEVBLK, IO$_READVBLK, IO$_SENSEMODE,
                            IO$_SETMODE;

  FROM ConditionHandlingProcedures IMPORT 
                                          LIB$SIGNAL;


  CONST
    maxbufsize = 256;       (* output is buffered to minimize QIO calls *)
    NULL = 0C;
    LF   = 12C;
    CR   = 15C;
    DEL  = 177C;

   (* values obtained from TTDEF and IODEF macros: *)
    TT_WRAP       = {9};
    TT_SYNC       = {5};
    IO_NOFORMAT   = {8};    (* WriteBuffer won't translate any FFs or VTs *)
    IO_TYPEAHDCNT = {6};    (* BusyRead checks type-ahead buffer *)
    IO_NOECHO     = {6};    (* BusyRead does a half-duplex read *)

  VAR
    buffer  : ARRAY [0..maxbufsize-1] OF CHAR;   (* output buffer *)
    buflen  : [0..maxbufsize]; (* number of characters in output buffer *)
    channel : SHORTWORD;        (* io channel assigned by VMS *)
    result  : CARDINAL;        (* status code returned from VMS *)
    iosb    : RECORD           (* io status block filled by QIO calls *)
                typeaheadcnt : SHORTWORD;
                termoffset : SHORTWORD;
                c2 : CARDINAL
              END;
    oldsetting,
    newsetting : ARRAY [1..2] OF BITSET;   (* for terminal characteristics *)


(******************************************************************************)

  PROCEDURE Read (VAR ch : CHAR);

  BEGIN
    result := SYS$QIOW(0,
                       channel,
                       BITSET(IO$_READVBLK),
                       ADR(iosb),
                       0,0,
                       ADR(ch),
                       1,
                       0,0,0,0);
(* DEBUG
IF NOT ODD(result) THEN LIB$SIGNAL(result) END;
GUBED *)
  END Read;

(******************************************************************************)

  PROCEDURE ReadString (VAR s : ARRAY OF CHAR);

(* SYSDEP: Read a string of characters.
   The routine is terminated upon CR, CTRL/Z, or an ESC sequence if line
   editing is enabled, otherwise upon 0..37C (except for LF, VT, FF, TAB, BS).
   Got that?!  Termination also occurs if s becomes full.
   If s is not full then NULLs are used to pad out the string.
   DEL can be used to erase the last character entered.
   Under version 4 of VMS, line editing is possible on a VT100 compatible VDU,
   but things go haywire if the cursor reaches the right margin.
*)

    VAR
      next : CARDINAL;

  BEGIN
    result := SYS$QIOW(0,
                       channel,
                       BITSET(IO$_READVBLK),
                       ADR(iosb),
                       0,0,
                       ADR(s),
                       HIGH(s)+1,   (* maximum size of s *)
                       0,
                       0,           (* default terminating character set *)
                       0,0);
(* DEBUG
IF NOT ODD(result) THEN LIB$SIGNAL(result) END;
GUBED *)
    next := CARDINAL(iosb.termoffset);   (* number of characters read *)
(* SYSDEP: pad s with NULLs if not full *)
    WHILE next <= HIGH(s) DO
      s[next] := NULL;
      INC(next);
    END;
  END ReadString;

(******************************************************************************)

  PROCEDURE BusyRead (VAR ch : CHAR) : BOOLEAN;

(* SYSDEP: Return TRUE iff ch is read (with no echo) from type-ahead buffer.
   If nothing in type-ahead buffer then ch is undefined and we return FALSE.
   Note that checking type-ahead buffer using IO$_SENSEMODE is faster than
   doing a timed read with timeout of 0.
*)

  BEGIN
    result := SYS$QIOW(0,
                       channel,
                       BITSET(IO$_SENSEMODE) + IO_TYPEAHDCNT,
                       0,0,0,
                       ADR(iosb),
                       0,0,0,0,0);
(* DEBUG
IF NOT ODD(result) THEN LIB$SIGNAL(result) END;
GUBED *)
    IF CARDINAL(iosb.typeaheadcnt) > 0 THEN
      (* char(s) in type-ahead buffer *)
      result := SYS$QIOW(0,
                         channel,
                         BITSET(IO$_READVBLK) + IO_NOECHO,
                         ADR(iosb),
                         0,0,
                         ADR(ch),1,   (* read only one char *)
                         0,0,0,0);
   (* DEBUG
   IF NOT ODD(result) THEN LIB$SIGNAL(result) END;
   GUBED *)
      RETURN TRUE;
    ELSE 
      RETURN FALSE;
    END;
  END BusyRead;

(******************************************************************************)

  PROCEDURE Write (ch : CHAR);

  BEGIN
    IF buflen = maxbufsize THEN
      WriteBuffer;
    END;
    buffer[buflen] := ch;
    INC(buflen);
  END Write;

(******************************************************************************)

  PROCEDURE WriteString (s: ARRAY OF CHAR);

    VAR
      i : INTEGER;

  BEGIN
(* SYSDEP: LEN assumes end of string is first NULL, or string is full *)
    FOR i := 0 TO LEN(s) - 1 DO
      Write(s[i]);
    END;
  END WriteString;

(******************************************************************************)

  PROCEDURE WriteInt (i : INTEGER);

(* We call WriteCard after writing any '-' sign. *)

  BEGIN
    IF i < 0 THEN
      Write('-');
      i := ABS(i);
    END;
    WriteCard(CARDINAL(i));
  END WriteInt;

(******************************************************************************)

  PROCEDURE WriteCard (c : CARDINAL);

(* Since the vast majority of given values will be small numbers, we avoid
   recursion until c >= 100.
*)

  BEGIN
    IF c < 10 THEN
      Write( CHR(ORD('0') + c) );
    ELSIF c < 100 THEN
      Write( CHR(ORD('0') + (c DIV 10)) );
      Write( CHR(ORD('0') + (c MOD 10)) );
    ELSE 
      WriteCard(c DIV 100);   (* recursive if c >= 100 *)
      c := c MOD 100;
      Write( CHR(ORD('0') + (c DIV 10)) );
      Write( CHR(ORD('0') + (c MOD 10)) );
    END;
  END WriteCard;

(******************************************************************************)

  PROCEDURE WriteLn;

  BEGIN
    Write(CR);
    Write(LF);
    WriteBuffer;
  END WriteLn;

(******************************************************************************)

  PROCEDURE WriteBuffer;

(* Output the buffer; either the buffer has filled or the user has explicitly
   requested the terminal to be updated.
*)

  BEGIN
    IF buflen > 0 THEN
      result := SYS$QIOW(0,
                         channel,
                         BITSET(IO$_WRITEVBLK) + IO_NOFORMAT,
                         ADR(iosb),
                         0,
                         0,
                         ADR(buffer),
                         buflen,
                         0,0,0,0);
   (* DEBUG
   IF NOT ODD(result) THEN LIB$SIGNAL(result) END;
   GUBED *)
      buflen := 0;
    END;
  END WriteBuffer;

(******************************************************************************)

  PROCEDURE RestoreTerminal;

  BEGIN
    result := SYS$QIOW(0,
                       channel,
                       IO$_SETMODE,
                       ADR(iosb),
                       0,0,
                       ADR(oldsetting),   (* restore terminal characteristics *)
                       8,                 (* size of oldsetting in bytes *)
                       0,0,0,0);
    IF NOT ODD(result) THEN
      LIB$SIGNAL(result)
    END;
  END RestoreTerminal;

(******************************************************************************)

  PROCEDURE Halt (status : CARDINAL);

  BEGIN
    WriteBuffer;
    RestoreTerminal;
(* SYSDEP: set high order bit so that no CLI message will be seen *)
    status := status + 10000000H;
    result := SYS$EXIT(status);
  END Halt;

(******************************************************************************)

BEGIN
  buflen := 0;
  result := SYS$ASSIGN("TT",channel,0,0);
  IF NOT ODD(result) THEN
    LIB$SIGNAL(result)
  END;
  result := SYS$QIOW(0,
                     channel,
                     IO$_SENSEMODE,
                     0,0,0,
                     ADR(oldsetting),   (* save terminal characteristics *)
                     8,                 (* size of oldsetting in bytes *)
                     0,0,0,0);
  IF NOT ODD(result) THEN
    LIB$SIGNAL(result)
  END;
  newsetting := oldsetting;
  newsetting[2] := newsetting[2] + TT_SYNC;
  newsetting[2] := newsetting[2] - TT_WRAP;
(* SYSDEP: set terminal /ttsync /nowrap *)
  result := SYS$QIOW(0,
                     channel,
                     IO$_SETMODE,
                     ADR(iosb),
                     0,0,
                     ADR(newsetting),   (* set new terminal characteristics *)
                     8,                 (* size of newsetting in bytes *)
                     0,0,0,0);
  IF NOT ODD(result) THEN
    LIB$SIGNAL(result)
  END;
END ScreenIO.
