#define VERSION "2.31"
/*
 *
 * AUTHOR(s)
 *     Mark Senn wrote the early versions of this program for the
 *     BBN BitGraph.  Stephan Bechtolsheim, Bob Brown, Richard
 *     Furuta, James Schaad and Robert Wells improved it.  Norm
 *     Hutchinson ported the program to the Sun.  Neal Holtz ported
 *     it to the Apollo, and from there to producing PostScript
 *     output. Scott Jones added intelligent font substitution.
 *     Piet van Oostrum added support for BSD4.1, the TEXFONTS
 *     environment variable, better positioning, resource (VM) limiting
 *     and PK files.
 *     Les Carr added the capacity for use of LaserWriter builtin fonts
 *     Piet van Oostrum merged the last two changes and rewrote the code for
 *     reading tfm files.
 */

/* Basic method:
 * Using the local font cacheing machinery that was in the previewer,
 * we can easily manage to send the bitmap for each chracter only once.
 * Two passes are made over each page in the DVI file.  The first pass
 * simply outputs the bitmaps for all characters on that page that haven't
 * been sent before.  The second pass outputs all the character setting
 * and positioning commands.  This allows us to bracket the setting portion
 * with PostScript save's and restore's, thus reclaiming considerable
 * virtual memory after each page.
 *
 * All coordinates are output in the PostScript system (TeX origin),
 * and in integer units of rasters (300/inch) -- except for character
 * widths, which are sent as floating point numbers.
 *
 * About 2 pages of PostScript support code must be sent to the LaserWriter
 * before this stuff goes.  It is automatically included unless the
 * -h option is given.
 */


/* Change log:
 *
 * Early 1985, (nmh) -- ported sun version to Apollo. 
 * A little later (nmh) -- changed to continue on in the event of missing
 *                      font files.
 * 30-Mar-85 (nmh) -- added -a option to specify a different PXL area
 * 30-Mar-85 (nmh) -- changed default PXL area to /pxl118
 * 31-Mar-85 (nmh) -- fixed bug in OpenFontFile() regarding more than MAXOPEN
 *                    PXL files -- changed to mark files as closed in font_entry.
 *  7-Apr-85 (nmh) -- made command line argument decoding case insensitive.
 *                    cleaned up handling of DVI file name argument.
 * 30-May-85 (nmh) -- new version; hacked to output PostScript commands
 *  6-Jun-85 (nmh) -- added relative positioning (20% smaller PostScript output)
 *                    add -m option to specify mag
 * 11-Jun-85 (nmh) -- fixed bug regarding char spacings in very long "words"
 * 12-Jun-85 (nmh) -- v1.02 - process DVI pages in reverse order
 * 13-Jun-85 (nmh) -- fixed bug re PXL files getting opened too often when no PreLoad
 * 14-Jun-85 (nmh) -- font dict created in PostScript only when 1st char of font downloaded
 *                    add -m0 -mh -m1 etc. to specify magsteps
 * 16-Aug-85 (nmh) -- added -c option t0 create output file in spool area (Apollo specific)
 *                    added -h option to copy header file to output
 *                    added -o option to pass options through to PostScript
 * 20-Aug-85 (nmh) -- v1.03
 * 24-Aug-85 (nmh) -- add -q option (for quiet operation).
 *                    changed -o option to check PostScript option
 *                    changed to output coordinates in TeX system (but
 *                    scaled to raster units) -- (0,0) at 1" in and down from
 *                      top left (better for use with different size paper).
 *                 -- v2.00
 * 25-Aug-85 (nmh) -- added dictionary enclosure to Tex.ps, and output
 *                      suitable prolog here.
 * 26-Aug-85 (nmh) -- changes to tex.ps to support Macintosh documents.
 * 14-Sep-85 (nmh) -- added keyword=value decoding to \special;
 * 15-Sep-85 (nmh) -- added -i file option.
 * 23-Sep-85 (saj) -- added font substitution for case when font is
 *                    unavailable at requested mag. (a frequent occurrence
 *                    with some macro packages like LaTeX)
 * 29-May-86 (pvo) -- modified to read PK files rather than pxl files
 * 02-Jun-86 (pvo) -- modified to limit VM usage for big characters
 * 10-Jul-86 (pvo) -- modified area handling to include TEXFONTS path
 * 10-Jul-86 (pvo) -- modified positioning to use Knuth's algorithm
				 (more or less)
 * 11-May-87 (pvo) -- general cleanup
 * 13-Mar-87 (lac) -- added -x option for Soton user
 * ??-Mar-87 (lac) -- added pstext \special keyword
 * 18-Mar-87 (lac) -- new version which uses built-in fonts on the Apple
 *		      LaserWriter and allows the use of psfig
 * 12-May-87 (pvo) -- merges (lac) and (pvo) changes
 * 14-May-87 (pvo) -- rewrote tfm file reading
 * 27-May-87 (pvo) -- changed SetString to check for rounding errors
 * 05-Jun-87 (pvo) -- include path search for fontmap and tfm files
 * 06-Nov-87 (pvo) -- enhanced the x option to include ranges.
 *   -May-88 (miw) -- Renamed x option to P;  x option does pages-per-page.
 *		      Changed special structure (added "nosave",
 *		      multiple psfile per special).
 *		      Added switches for resolution -R, budget -b, A5 -A,
 *		      printing VM (-V), extra lw+ fonts -E.
 *		      Fixed bug in FNT1 (not skipping extra byte).
 *		      Removed reencoding of PostScript fonts.
 * 02-Jun-88 (miw) -- Added page-separation option (-F)
 */


/**********************************************************************/
/************************  Global Definitions  ************************/
/**********************************************************************/

/* This version purports to drive a PostScript device (slowly) */


typedef int BOOLEAN;
#define NEW(A) ((A *) malloc(sizeof(A)))
#define abs(x) ((x)<0?-(x):x)
#define DEBUG   1			/* for massive printing of input */
					/* trace information; select by -d */
					/* option after filename: */
					/* dviview filename -d */
#ifdef DEBUG
int Debug = 0;
#endif
                          /* to enable statistics reporting via -s option */
#define STATS

#define BINARYOPEN fopen		/* byte-oriented host version */

#define ARITHRSHIFT 1                   /* define if ">>" operator is a */
                                        /* sign-propagating arithmetic  */
                                        /*   right shift */
#define USEGLOBALMAG 1			/* when defined, the dvi global */
   					/*   magnification is applied   */
      
/* We can leave USEGLOBALMAG undefined when we have a limited
   number of font magnifications (at 300dpi) available.  Otherwise, we
   will simply complain about missing PXL files
 */

/* #undef USEGLOBALMAG */

                        /* define for "optimal" relative postioning, rather
                           than absolute.  Relative can reduce size of postcript
                           output 20% (and reduce print time by almost as much */
#define USERELPOS 1 

#define  DVIFORMAT        2
#define  TRUE             1
#define  FALSE            0
#define  FIRSTPXLCHAR     0
#define  LASTPXLCHAR    255

#ifndef FONTAREA
#define  FONTAREA         "/usr/lib/tex/fonts"
#endif

#ifndef PXLAREA
#define	PXLAREA		FONTAREA
#endif

#ifdef BUILTIN
#ifndef FONTMAP
			/* directory comes from FONTAREA or TEXFONTS or -a */
#define FONTMAP		"TeXPSfonts.map"
#endif

#define MAXBUILTINS	100
#endif

#ifndef HDRFILE
#define HDRFILE         "/usr/lib/tex.ps"
#endif

#ifdef apollo
#ifndef SPOOLFILE
#define SPOOLFILE       "/local/spool/laser/tex."
#endif
#define MAXFLEN 28
#endif apollo

#ifdef apollo
#define  MAXOPEN         45  /* limit on number of open PXL files */
#else !apollo
#define  MAXOPEN         12  /* limit on number of open PXL files */
#endif

#define  NPXLCHARS      256
#ifndef PK
#define  PXLID         1001
#else
#define  PKID         89
#define  PKXXX1	240
#define  PKXXX2	241
#define  PKXXX3 242
#define  PKXXX4 243
#define  PKYYY	244
#define  PKPOST	245
#define  PKNOOP 246
#define  PKPRE  247
#endif /* PK */
#define  READ             4  /* for access() */
#ifndef RESOLUTION
#define  RESOLUTION      300
#endif
#ifndef hconvRESOLUTION
#define  hconvRESOLUTION resolution
#endif
#ifndef vconvRESOLUTION
#define  vconvRESOLUTION resolution
#endif
#define  STACKSIZE      100
#define  STRSIZE        257
#define  MAX_PATHS	20
#define  NONEXISTANT     -1   /* offset for PXL files not found */
#define  NO_FILE        (FILE *)-1
#define  UNDEFPOS	-314159728
#define max_drift	2	/* max nbr of pixels drift by rounding */

#ifdef apollo           /* define for enabling of -c option (create output file) */
#define CREOPT
#endif apollo

/**********************************************************************/
/***********************  external definitions  ***********************/
/**********************************************************************/

#include "commands.h"
#include <sys/param.h>
#include <signal.h>
#include <stdio.h>
#include "findfile.h"
#include <ctype.h>

int  access();
char *index();
char *malloc();
char *getenv();
int  free();
char *rindex();
char *sprintf();
char *strcpy(); 
char *logname();
#ifdef PK
int  pk_packed_num();
int  pk_get_nyb();
#endif /* PK */

#define EQ(a,b) (strcmp(a,b)==0)

                        /* output a formatted string */
#define EMIT      fprintf
                        /* output a simple string */
#define EMITS(s)  fputs(s,outfp)
                        /* output an escaped octal number */
#define EMITO(c)  PutOct(c)
                        /* output a decimal integer */
#define EMITN(n)  PutInt(n)
                        /* output a byte value in Hex */
#define EMITH(h)  putc(*(digit+((h>>4)&0xF)),outfp),\
                  putc(*(digit+(h&0xF)),outfp)
                        /* output a single character */
#define EMITC(c)  putc(c,outfp)

#define OUTFP	(prolog?prolog:outfp)
#define GEMIT	EMIT
#define GEMITH(h)  putc(*(digit+((h>>4)&0xF)),OUTFP),\
                  putc(*(digit+(h&0xF)),OUTFP)
#define GEMITS(s)  fputs(s,OUTFP)
                                   /* formatted i/o was killing us, so build some tables */
char    *digit = "0123456789ABCDEF";

int dummyInt;
short dummyShort;
char dummyChar;
#ifdef PK
int bit_weight = 0, input_byte;
int rep_count;
int dyn_f;
int power[] = {01,02,04,010,020,040,100,200};
int gpower[] = {00,01,03,07,017,037,077,0177,0377};
#endif /* PK */

/* definitions for loading characters */

#define NOTLOADED 0
#define PERMANENT 1
#define TEMPORARY 2

#ifdef BUDGET
#define fontcost(p) 2372
#define charcost(ptr) (((ptr->width + 7) >> 3) * ptr->height+70)
#define BIGCHAR(ptr) (charcost(ptr)>budget/100)
int budget = BUDGET;
#endif

/**********************************************************************/
/*************************  Global Procedures  ************************/
/**********************************************************************/

/* Note: Global procedures are declared here in alphabetical order, with
   those which do not return values typed "void".  Their bodies occur in
   alphabetical order following the main() procedure.  The names are
   kept unique in the first 6 characters for portability. */

void	AbortRun();
float	ActualFactor();
void	AllDone();
FILE*	BINARYOPEN();
int     ChkOpt();               /* Check PostScript option for validity */
void    CopyFile();
void    DecodeArgs();
void    DoSpecial();
void    EmitChar();
void	Fatal();
int	FindPath();
void	FindPostAmblePtr();
void	GetBytes();
void	GetFontDef();
char   *GetKeyStr();
int     GetKeyVal();
int     HasBeenRead();
int	InPageList();
#ifdef BUILTIN
int	IsBuiltIn();
#endif
int     IsSame();
void    lcase();
void	MoveDown();
void	MoveOver();
void	MoveRight();
int     NoSignExtend();         /* see cautionary note in code, re arithmetic vs logical shifts */
void	OpenFontFile();
#ifdef CREOPT
FILE*   OpenOutput();
#endif CREOPT
int	PixRound();
#ifdef BUILTIN
char   *PSname();
#endif
void    PutInt();
int	ReadFontDef();
#ifdef BUILTIN
/** void	ReadMap(); **/
#endif
void	ReadPostAmble();
void	SetChar();
void	SetFntNum();
void    SetPosn();
void	SetRule();
void    SetString();
int     SignExtend();           /* see cautionary note in code, re arithmetic vs logical shifts */
void	SkipFontDef();
int	SplitPath();
void	Warning();


/**********************************************************************/
/***********************  Font Data Structures  ***********************/
/**********************************************************************/

struct char_entry {		/* character entry */
   unsigned short width, height;/* width and height in pixels */
   short xOffset, yOffset;      /* x offset and y offset in pixels */
#ifdef PK
   char dyn_f;			/* dyn_f value */
   char turn_on;
#endif /* PK */
   struct {
       int isloaded;		/* 0 = not loaded,1 = permanent,2= temporary */
       union {
	   int fileOffset;
	   char *pixptr; } address;
       } where;
   int tfmw;			/* TFM width */
   short dx;			/* hor escapement in pixels */
   };

struct font_entry {  /* font entry */
   int k, c, s, d, a, l;
   char area[STRSIZE];	/* FNT_DEF command parameters  */
   char n[STRSIZE];	/* FNT_DEF command parameters  */
   int font_space;	/* computed from FNT_DEF s parameter        */
   int font_mag;	/* computed from FNT_DEF s and d parameters */
   char psname[STRSIZE]; /* PostScript name of the font             */
   char name[STRSIZE];	/* full name of PXL file                    */
   FILE *font_file_id;  /* file identifier (NO_FILE if none)         */
#ifdef BUILTIN
   int builtin;	/* is it a builin font? */
#endif
   int magnification;	/* magnification read from PXL file         */
   int designsize;	/* design size read from PXL file           */
   struct char_entry ch[NPXLCHARS];/* character information         */
   struct font_entry *next;
   int ncdl;            /* # of different chars actually downloaded */
#ifdef STATS
   int nbpxl;           /* # of bytes of PXL data downloaded        */
   int ncts;            /* total # of characters typeset */
#endif
   };

struct pixel_list
{
    FILE *pixel_file_id;        /* file identifier                          */
    int use_count;              /* count of "opens"                         */
    };



/**********************************************************************/
/*************************  Global Variables  *************************/
/**********************************************************************/

int   FirstPage = -1000000;     /* first page to print (uses count0)   */
int   LastPage = 1000000;       /* last page to print                    */

/* We need to remember which pages the user would like.  We build a linked
   list that allows us to decide (for any given page) whether it should
   be included in the output file.
*/
struct pagelist {
    struct pagelist *pl_alt;	/* next in a series of alternates */
	int ps_low, ps_high;	/* allowed range */
} *PageList = 0;	/* the list of allowed pages */

char filename[STRSIZE];         /* DVI file name			   */

int   G_create = FALSE;         /* create an output file in spool area ?   */
int   G_errenc = FALSE;	        /* has an error been encountered?          */
int   G_header = TRUE;          /* copy header file to output?             */
char  G_Logname[STRSIZE];       /* name of log file, if created            */
int   G_interactive = FALSE;    /* is the program running interactively    */
                                /* (i.e., standard output not redirected)? */
int   G_logging = 0;            /* Are we logging warning messages?        */
int   G_logfile = FALSE;        /* Are these messages going to a log file? */
FILE *G_logfp;                  /* log file pointer (for errors)           */
char  G_progname[STRSIZE];      /* program name                            */
int   G_quiet = FALSE;          /* for quiet operation                     */
int   G_nowarn = FALSE;         /* don't print out warnings                */

int   hconv, vconv;		/* converts DVI units to pixels             */
int   den;			/* denominator specified in preamble        */
FILE *dvifp  = NULL;		/* DVI file pointer                         */
int   PreLoad = TRUE;	        /* preload the font descriptions?	     */
struct font_entry *prevfont=NULL;  /* font_entry pointer, previous character */
struct font_entry *fontptr;     /* font_entry pointer                       */
struct font_entry *hfontptr=NULL;/* font_entry pointer                      */
char *Ifile[100];               /* files to include */
int   h;			/* current horizontal position              */
int   hh;			/* idem in pixels		              */
int   hhh = UNDEFPOS;           /* current h on device */
int   v;			/* current vertical position                */
int   vv;			/* idem in pixels		              */
int   vvv = UNDEFPOS;           /* current v on device */
int   mag;			/* magnification specified in preamble      */
int   ncopies = 1;              /* number of copies to print                */
int   pages_per_page = 1;       /* number of TeX pages per LaserWriter page*/
int   ndone = 0;                /* number of pages converted */
int   nif = 0;                  /* number of files to include */
int   nopen;			/* number of open PXL files                 */
int   nps = 0;                  /* number of PostScript commands to send    */
int   num;			/* numerator specified in preamble          */
BOOLEAN A5 = FALSE;		/* printing A5 pages */
BOOLEAN printVMstats = FALSE;	/* output of statistics at end of job */
BOOLEAN extrafonts = FALSE;	/* use extra fonts on LW+ */
int resolution = RESOLUTION;	/* resolution of device (in ppi) */
#ifdef CREOPT
char  outfname[256];            /* name of output file                      */
#endif CREOPT
FILE  *outfp = NULL;            /* output file                              */
struct font_entry *pfontptr = NULL; /* previous font_entry pointer          */
struct pixel_list pixel_files[MAXOPEN+1];
                                /* list of open PXL file identifiers        */
long  postambleptr;		/* Pointer to the postamble                 */
FILE *pxlfp;			/* PXL file pointer                         */
char *PXLpath[MAX_PATHS];	/* PXL path names for search		    */
int   PXLpathlen;		/* nbr of PXL path names for search	    */
#ifdef BUILTIN
char *TFMpath[MAX_PATHS];	/* TFM path names for search		    */
int   TFMpathlen;		/* nbr of TFM path names for search	    */
#endif
long  ppagep;	                /* previous page pointer		     */
char *PScmd[100];               /* PostScript commands to send              */
int   Reverse = TRUE;         /* process DVI pages in reverse order ? */
char  rootname[STRSIZE];      /* DVI filename without extension */
#ifdef BUILTIN
char *TFMpath[MAX_PATHS];	/* TFM path names for search		    */
int   TFMpathlen;		/* nbr of TFM path names for search	    */
struct BuiltIn {
  struct BuiltIn *next;
  char TeXname[STRSIZE];	/* TeX Name of the LaserWriter built-in font */
  char Postscript[STRSIZE];	/* Internal Name of the built-in font */
} *BuiltIns = 0;
char *prefix;			/* prefix for prolog, page and epilog files */
char prolog_name[STRSIZE], epilog_name[STRSIZE],
  page_name[STRSIZE];
FILE *prolog, *epilog;

#endif

#ifdef STATS
int   Stats = FALSE;          /* are we reporting stats ?                 */
int   Snbpxl = 0;             /* # of bytes of pixel data                 */
int   Sonbpx = 0;             /* "optimal" number of bytes of pixel data  */
int   Sndc = 0;               /* # of different characters typeset        */
int   Stnc = 0;               /* total # of chars typeset                 */
int   Snbpx0, Sndc0, Stnc0;   /* used for printing incremental changes per dvi page */
#endif

int usermag = 0;              /* user specified magnification */

/**********************************************************************/
/*******************************  main  *******************************/
/**********************************************************************/

main(argc, argv)
int argc;
char *argv[];

{
    struct stack_entry {  /* stack entry */
	int h, v, w, x, y, z, hh, vv;  /* what's on stack */
    };


    int command;	    /* current command			     */
    int count[10];          /* the 10 counters at begining of each page */
    long cpagep;	    /* current page pointer		     */
    int  Emitting = FALSE;  /* are we outputting typsetting instructions? */
    register int i;	    /* command parameter; loop index	     */
    int k;		    /* temporary parameter		     */
    char n[STRSIZE];	    /* command parameter		     */
    int PassNo = 0;         /* which pass over the DVI page are we on?  */
    int SkipMode = FALSE;   /* in skip mode flag                     */
    int sp;		    /* stack pointer			     */
    struct stack_entry stack[STACKSIZE];   /* stack		     */
    int t;		    /* temporary			     */
    char SpecialStr[STRSIZE]; /* "\special" strings                  */
    register struct char_entry *tcharptr; /* temporary char_entry ptr*/
    int val, val2;          /* temporarys to hold command information*/
    int w;		    /* current horizontal spacing	     */
    int x;		    /* current horizontal spacing	     */
    int y;		    /* current vertical spacing		     */
    int z;		    /* current vertical spacing		     */

    nopen = 0;
    strcpy(G_progname, argv[0]);

    *PXLpath = getenv ("TEXFONTS");
    if (! *PXLpath || ! **PXLpath) {
      *PXLpath = PXLAREA;       /* default pixel area */
      *TFMpath = FONTAREA;      /* default font area */
    }
    else *TFMpath = *PXLpath;

    DecodeArgs( argc, argv );
#ifdef BUILTIN
    ReadMap();
#endif

#ifdef apollo
    set_sbrk_size( 2048*1024 );
#endif

    if ((i = NoSignExtend(dvifp, 1)) != PRE)  {
	fprintf(stderr,"\n");
	Fatal("%s: PRE doesn't occur first--are you sure this is a DVI file?\n\n",
	G_progname);
    }

    i = SignExtend(dvifp, 1);
    if (i != DVIFORMAT)  {
	fprintf(stderr,"\n");
	Fatal("%s: DVI format = %d, can only process DVI format %d files\n\n",
	G_progname, i, DVIFORMAT);
    }

#ifdef CREOPT
    if( G_create )
        outfp = OpenOutput();
    else
#endif CREOPT

      if (prolog)
        outfp = prolog;
      else
	outfp = stdout;

/* it is important that these be the very first things output !!! */

/* we know that outfp is the prolog, so no need to use GEMIT */
    EMIT(outfp, "%%!\n");
    EMIT(outfp, "%%%%Title: %s\n", filename);
    EMIT(outfp, "%%%%Creator: %s\n", G_progname);

    if( G_header )
        CopyFile( HDRFILE );

    for( i=0; i<nif; i++ )      /* copy all included files */
        CopyFile( Ifile[i] );

    EMIT(outfp, "TeXDict begin \n");
    if (A5)
      EMIT(outfp, "A5pages\n");
    else
      EMIT(outfp, "%d @TexPagesPerPage\n", pages_per_page);
    if (printVMstats)
      EMIT(outfp, "/printVMstats true def\n");
    if (extrafonts)
      EMIT(outfp, "makeLWPlusFonts\n");
    EMIT(outfp, "@start\n");

    for( i=0; i<nps; i++ )      /* prefix valid PostScript options with a "@" */
        if( ChkOpt(PScmd[i]) )
            EMIT(outfp, "@%s\n", PScmd[i]);
        else
            Fatal( "%s is an invalid option\n", PScmd[i] );

    if( ncopies > 1 )
        EMIT(outfp, "%d @copies\n", ncopies);

    EMIT(outfp, "%%%%EndProlog\n");

    if (Reverse) {
        ReadPostAmble(PreLoad);
        fseek(dvifp, ppagep, 0);
        }
    else {
        if (PreLoad) {
            ReadPostAmble(TRUE);
            fseek(dvifp, (long) 14, 0);
            }
        else {
    	    num = NoSignExtend(dvifp, 4);
    	    den = NoSignExtend(dvifp, 4);
    	    mag = NoSignExtend(dvifp, 4);
#ifdef USEGLOBALMAG
            if( usermag > 0 && usermag != mag )
                fprintf(stderr, "DVI magnification of %d over-ridden by user mag of %d\n", mag, usermag );
#endif
            if( usermag > 0 ) mag = usermag;
#ifndef USEGLOBALMAG
            if( mag != 1000 ) fprintf(stderr, "Magnification of %d ignored.\n", mag);
#endif
    	    hconv = DoConv(num, den, hconvRESOLUTION);
    	    vconv = DoConv(num, den, vconvRESOLUTION);
            }
        k = NoSignExtend(dvifp, 1);
        GetBytes(dvifp, n, k);
        }

    PassNo = 0;
    
    while (TRUE)

	switch (command=NoSignExtend(dvifp, 1))  {

	case SET1:case SET2:case SET3:case SET4:
	    val = NoSignExtend(dvifp, command-SET1+1);
	    if (!SkipMode) SetChar(val, command, PassNo);
	    break;

	case SET_RULE:
	    val = NoSignExtend(dvifp, 4);
	    val2 = NoSignExtend(dvifp, 4);
            if (Emitting) SetRule(val, val2, 1);
	    break;

	case PUT1:case PUT2:case PUT3:case PUT4:
	    val = NoSignExtend(dvifp,command-PUT1+1);
	    if (!SkipMode) SetChar(val, command, PassNo);
	    break;

	case PUT_RULE:
	    val = NoSignExtend(dvifp, 4);
	    val2 = NoSignExtend(dvifp, 4);
            if (Emitting) SetRule(val, val2, 0);
	    break;

	case NOP:
	    break;

	case BOP:
	    cpagep = ftell(dvifp) - 1;
	    for (i=0; i<=9; i++)
		count[i] = NoSignExtend(dvifp, 4);
	    ppagep = NoSignExtend(dvifp, 4);

	    h = v = w = x = y = z = 0;
            hhh = vvv = UNDEFPOS;
	    hh = vv = 0;
	    sp = 0;
	    fontptr = NULL;
            prevfont = NULL;

	    if(InPageList(count[0]))
		SkipMode= FALSE;
            else if( count[0] < FirstPage || count[0] > LastPage) 
                	SkipMode = TRUE;
	    else
                	SkipMode = FALSE;

            Emitting = (PassNo != 0) && !SkipMode;

            if( !SkipMode ) {
                if( PassNo == 0) {
			if (prolog) {
				sprintf(page_name, "%s%d", prefix, count[0]);
				if (outfp= fopen(page_name, "w"), outfp == NULL)
				  Fatal("can't open %s\n", page_name);
			}
                        EMIT(outfp,"%d @bop0\n", count[0]);
#ifdef STATS
                        if( Stats ) {
                                Sndc0 = Sndc;
                                Stnc0 = Stnc;
                                Snbpx0 = Snbpxl;
                                }
#endif
                        if( !G_quiet ) 
                                fprintf(stderr, "[%d", count[0] );
                        }
                else
                        EMIT(outfp,"%d @bop1\n", count[0]);
                }
	    break;

	case EOP:
            if( !SkipMode ) {
                if( PassNo == 0 ) {     /* start second pass on current page */
                    fseek(dvifp,cpagep,0);
                    PassNo = 1;
                    }
                else {                  /* end of second pass, and of page processing */
                    EMIT(outfp,"@eop\n");
		    if (prolog)
		      fclose(outfp);
#ifdef STATS
                    if( Stats )
                        fprintf(stderr," - %d total ch,  %d diff ch,  %d pxl bytes]\n",
                                Stnc-Stnc0, Sndc-Sndc0, Snbpxl-Snbpx0);   
                    else
#endif
                        if( !G_quiet ) {
                                fprintf(stderr,"] ");
                                if( (++ndone % 10) == 0 ) fprintf(stderr,"\n");
                                }
                    PassNo = 0;
                    }
                }
            else
                PassNo = 0;
            if( PassNo == 0 && Reverse ) { 
                if( ppagep > 0 )
                    fseek(dvifp, ppagep, 0);
                else 
                    AllDone();
                }
	    break;

	case PUSH:
	    if (sp >= STACKSIZE)
		Fatal("stack overflow");
	    stack[sp].h = h;
	    stack[sp].v = v;
	    stack[sp].w = w;
	    stack[sp].x = x;
	    stack[sp].y = y;
	    stack[sp].z = z;
	    stack[sp].hh = hh;
	    stack[sp].vv = vv;
	    sp++;
	    break;

	case POP:
	    --sp;
	    if (sp < 0)
		Fatal("stack underflow");
	    hh = stack[sp].hh;
	    vv = stack[sp].vv;
	    h = stack[sp].h;
	    v = stack[sp].v;
	    w = stack[sp].w;
	    x = stack[sp].x;
	    y = stack[sp].y;
	    z = stack[sp].z;
	    break;

	case RIGHT1:case RIGHT2:case RIGHT3:case RIGHT4:
            val = SignExtend(dvifp,command-RIGHT1+1);
	    if (Emitting) MoveOver(val);
	    break;

	case W0:
            if (Emitting) MoveOver(w);
	    break;

	case W1:case W2:case W3:case W4:
	    w = SignExtend(dvifp,command-W1+1);
            if (Emitting) MoveOver(w);
	    break;

	case X0:
            if (Emitting) MoveOver(x);
	    break;

	case X1:case X2:case X3:case X4:
	    x = SignExtend(dvifp,command-X1+1);
	    if (Emitting) MoveOver(x);
	    break;

	case DOWN1:case DOWN2:case DOWN3:case DOWN4:
            val = SignExtend(dvifp,command-DOWN1+1);
	    if (Emitting) MoveDown(val);
	    break;

	case Y0:
            if (Emitting) MoveDown(y);
	    break;

	case Y1:case Y2:case Y3:case Y4:
	    y = SignExtend(dvifp,command-Y1+1);
            if (Emitting) MoveDown(y);
	    break;

	case Z0:
            if (Emitting) MoveDown(z);
	    break;

	case Z1:case Z2:case Z3:case Z4:
	    z = SignExtend(dvifp,command-Z1+1);
	    if (Emitting) MoveDown(z);
	    break;

	case FNT1:case FNT2:case FNT3:case FNT4:
	    k = NoSignExtend(dvifp,command-FNT1+1);
            if (!SkipMode) {
		    SetFntNum(k, Emitting);
	    }
	    break;

	case XXX1:case XXX2:case XXX3:case XXX4:
	    k = NoSignExtend(dvifp,command-XXX1+1);
            GetBytes(dvifp, SpecialStr, k);
            if(Emitting) DoSpecial(SpecialStr, k);
	    break;

	case FNT_DEF1:case FNT_DEF2:case FNT_DEF3:case FNT_DEF4:
            k = NoSignExtend(dvifp, command-FNT_DEF1+1);
	    if (PreLoad || HasBeenRead(k) )
	    {
		SkipFontDef (k);
	    }
	    else
	    {
		ReadFontDef (k);
	    }
	    break;

	case PRE:
	    Fatal("PRE occurs within file");
	    break;

	case POST:
            AllDone();
	    break;

	case POST_POST:
 	    Fatal("POST_POST with no preceding POST");
	    break;

	default:
	    if (command >= FONT_00 && command <= FONT_63)
		{if (!SkipMode)
                     SetFntNum(command - FONT_00, Emitting);}
	    else if (command >= SETC_000 && command <= SETC_127)
		{if (!SkipMode) SetString(command, PassNo);}
	    else
		Fatal("%d is an undefined command", command);
	    break;

	}

}

/*-->AbortRun*/
/**********************************************************************/
/***************************  AbortRun  *******************************/
/**********************************************************************/

void
AbortRun(code)
int code;
{
    exit(code);
}


/*-->ActualFactor*/
/**********************************************************************/
/**************************  ActualFactor  ****************************/
/**********************************************************************/

float		/* compute the actual size factor given the approximation */
ActualFactor(unmodsize)
int unmodsize;  /* actually factor * 1000 */
{
    float realsize;	/* the actual magnification factor */

    realsize = (float)unmodsize / 1000.0;
    /* a real hack to correct for rounding in some cases--rkf */
    if(unmodsize==1095) realsize = 1.095445;	/*stephalf*/
    else if(unmodsize==1315) realsize=1.314534;	/*stepihalf*/
    else if(unmodsize==1577) realsize=1.577441;	/*stepiihalf*/
    else if(unmodsize==1893) realsize=1.892929;	/*stepiiihalf*/
    else if(unmodsize==2074) realsize=2.0736;	/*stepiv*/
    else if(unmodsize==2488) realsize=2.48832;  /*stepv*/
    else if(unmodsize==2986) realsize=2.985984;	/*stepiv*/
    /* the remaining magnification steps are represented with sufficient
	   accuracy already */
    return(realsize);
}


/*-->AllDone*/
/**********************************************************************/
/****************************** AllDone  ******************************/
/**********************************************************************/

void
AllDone()
{
    char t;
    struct font_entry *p;
    int per;

    if (epilog)
      outfp= epilog;

    EMIT(outfp,"@end\n");
    if( !G_quiet ) fprintf(stderr,"\n");

#ifdef CREOPT
    if( G_create ) {
        fclose(outfp);
        if( !G_quiet ) fprintf(stderr, "Output written on \"%s\"\n", outfname );
        }
#endif CREOPT

    if (G_errenc && G_logging == 1 && G_logfile)  {
	fseek(G_logfp, 0, 0);
	while ((t=getc(G_logfp)) != EOF)
	    putchar(t);
    }
    if (G_logging == 1 && G_logfile) printf("Log file created\n");

#ifdef STATS
    if (Stats) {
        fprintf( stderr, "Total chars   diff chars   pxl bytes\n" );
        fprintf( stderr, "      #   %%        #   %%       #   %%\n" );
        fprintf( stderr, "------- ---   ------ ---   ----- ---\n" );
        for( p=hfontptr; p!=NULL; p=p->next ) {
                fprintf( stderr, "%7d%4d", p->ncts, (100*p->ncts + Stnc/2)/Stnc );
                fprintf( stderr, "%9d%4d", p->ncdl, (100*p->ncdl + Sndc/2)/Sndc );
                fprintf( stderr, "%8d%4d", p->nbpxl, (100*p->nbpxl + Snbpxl/2)/Snbpxl );
                fprintf( stderr, "  %s\n", p->psname );
                }
        fprintf(stderr, "\nTotal number of characters typeset: %d\n", Stnc);
        fprintf(stderr, "Number of different characters downloaded: %d\n", Sndc);
        fprintf(stderr, "Number of bytes of pxl data downloaded: %d\n", Snbpxl);
        fprintf(stderr, "Optimal # of bytes of pxl data: %d\n", Sonbpx);
        }
#endif

    AbortRun(G_errenc);
}


/*-->ChkOpt*/   /* check a user supplied option for validity */
/*********************************************************************/
/****************************** ChkOpt *******************************/
/*********************************************************************/

#define ISOPT(s) if( EQ(str,s) ) return( TRUE )

int
ChkOpt( str )
char    *str;
{
/*        lcase(str);  doesn't work */

        ISOPT("note");          /* its a shame to build this into the program */
        ISOPT("letter");
        ISOPT("legal");
        ISOPT("landscape");
        ISOPT("manualfeed");
        return( FALSE );
}


/*-->CopyFile*/   /* copy a file straight through to output */
/*********************************************************************/
/***************************** CopyFile ******************************/
/*********************************************************************/

void
CopyFile( str )
char    *str;
{
        FILE    *spfp;
        char    t;

        if( (spfp=fopen(str,"r")) == NULL ) {
                fprintf(stderr,"Unable to open file %s\n", str );
                return;
                }
        if( !G_quiet ) fprintf(stderr," [%s", str);
        while( (t = getc(spfp)) != EOF ) {
                if (t >= '\010') EMITC(t); /* Don't transmit controls */
                }              
        fclose(spfp);
        if( !G_quiet ) fprintf(stderr,"]");
}


/*-->DecodeArgs*/
/*********************************************************************/
/***************************** DecodeArgs ****************************/
/*********************************************************************/

void
DecodeArgs( argc, argv )
int argc;
char *argv[];
{
    int argind;             /* argument index for flags              */
    char curarea[STRSIZE];  /* current file area		     */
    char curname[STRSIZE];  /* current file name		     */
    char *tcp, *tcp1;	    /* temporary character pointers	     */

    argind = 1;
    while (argind < argc) {
	tcp = argv[argind];
        if (*tcp == '-')
	    switch(*++tcp) {

                case 'a':       /* a selects different pxl font area */
                    *PXLpath = *TFMpath = argv[++argind];
                    break;
#ifdef CREOPT
                case 'c':       /* create an output file in spool area */
                    G_create = TRUE;
                    break;
#endif CREOPT
#ifdef DEBUG
		case 'd':	/* d selects Debug output */
		    Debug = TRUE;
		    break;
#endif
                case 'f':       /* next arg is starting pagenumber */
                    if( ++argind >= argc || sscanf(argv[argind], "%d", &FirstPage) != 1 )
                        Fatal("Argument is not a valid integer\n", 0);
                    break;

                case 'h':       /* don't copy PostScript header file through to output */
                    G_header = FALSE;
                    break;

                case 'i':       /* next arg is a PostScript file to copy */
                    if( ++argind >= argc )
                        Fatal("No argument following -i\n", 0);
                    Ifile[nif++] = argv[argind];
                    break;

		case 'l':	/* l prohibits logging of errors */
		    G_logging = -1;
		    break;
#ifdef USEGLOBALMAG
                case 'm':       /* specify magnification to use */
                    switch( (isupper(*++tcp)? tolower(*tcp): *tcp)) {

                    case '\0':       /* next arg is a magnification to use */
                        if( ++argind >= argc || sscanf(argv[argind], "%d", &usermag) != 1 )
                            Fatal("Argument is not a valid integer\n", 0);
                        break; 
                    case '0': usermag = 1000; break;
                    case 'h': usermag = 1095; break;
                    case '1': usermag = 1200; break;
                    case '2': usermag = 1440; break;
                    case '3': usermag = 1728; break;
                    case '4': usermag = 2074; break;
                    case '5': usermag = 2488; break;
                    default: Fatal("%c is a bad mag step\n", *tcp);
                    }
                    break;
#endif
                case 'n':       /* next arg is number of copies to print */
                    if( ++argind >= argc || sscanf(argv[argind], "%d", &ncopies) != 1 )
                        Fatal("Argument is not a valid integer\n", 0);
                    break;    

	    	case 'x':       /* next arg is number of TeX pages per LaserWriter page */
                    if( ++argind >= argc || sscanf(argv[argind], "%d", &pages_per_page) != 1 )
                        Fatal("Argument is not a valid integer\n", 0);
		    if (pages_per_page != 1)
		      	Reverse = !Reverse;	/* forward makes more sense here */
		    A5 = FALSE;
                    break;    
                case 'o':       /* next arg is a PostScript command to send */
                    if( ++argind >= argc )
                        Fatal("No argument following -o\n", 0);
                    PScmd[nps++] = argv[argind];
                    break;

		case 'p':	/* p prohibits pre-font loading */
		    PreLoad = 0;
                    Reverse = FALSE;    /* must then process in forward order */
		    break;

                case 'q':       /* quiet operation */
                    G_quiet = TRUE;
                    break;

                case 'r':       /* don't process pages in reverse order */
                    Reverse = !Reverse;
                    break;
#ifdef STATS                   
                case 's':       /* print some statistics */
                    Stats = TRUE;
                    break;
#endif
                case 't':       /* next arg is ending pagenumber */
                    if( ++argind >= argc || sscanf(argv[argind], "%d", &LastPage) != 1 )
                        Fatal("Argument is not a valid integer\n", 0);
                    break;

                case 'w':       /* don't print out warnings */
                    G_nowarn = TRUE;
                    break;

		case 'P':	/* next argument is a list of page numbers
				   for printing */
		    if( ++argind >=argc || ParsePages(argv[argind] ))
			Fatal("Argument is not a valid pagelist\n", 0);
		    FirstPage = -1000000;
		    LastPage = -1000000;
		    break;

		case 'A':	/* print two A5 pages on each A4 page */
		    A5 = TRUE;
		    pages_per_page = 2;
		    Reverse = !Reverse;
		    break;
		    
	        case 'V':	/* print VM stats */
		    printVMstats = TRUE;
		    break;

	        case 'E':	/* use LW+ fonts */
		    extrafonts = TRUE;
		    break;

	    	case 'R':       /* next arg is resolution (in ppi) */
                    if( ++argind >= argc || sscanf(argv[argind], "%d", &resolution) != 1 )
                        Fatal("Argument is not a valid integer\n", 0);
                    break;    

	    	case 'b':       /* next arg is VM budget */
                    if( ++argind >= argc || sscanf(argv[argind], "%d", &budget) != 1 )
                        Fatal("Argument is not a valid integer\n", 0);
                    break;    

	    	case 'F':	/* put header, pages and trailer in separate files */
		    if ( ++argind >= argc )
		      	Fatal("no prefix provided\n", 0);
		    prefix = argv[argind];
		    sprintf(prolog_name, "%spro", prefix);
		    sprintf(epilog_name, "%sepi", prefix);
		    
		    if (prolog= fopen(prolog_name, "w"), prolog == NULL)
		      Fatal("can't open %s\n", prolog_name);

		    if (epilog= fopen(epilog_name, "w"), epilog == NULL)
		      Fatal("can't open %s\n", epilog_name);
		    break;

		default:
		    printf("%c is not a legal flag\n", *tcp);
		}

        else {

            tcp = rindex(argv[argind], '/');    /* split into directory + file name */
            if (tcp == NULL)  {
        	curarea[0] = '\0';
        	tcp = argv[argind];
                }
            else  {
        	strcpy(curarea, argv[argind]);
                curarea[tcp-argv[argind]+1] = '\0';
        	tcp ++;
                }
        
            strcpy(curname, tcp);
            tcp1 = rindex(tcp, '.');   /* split into file name + extension */
            if (tcp1 == NULL) {
                strcpy(rootname, curname);
                strcat(curname, ".dvi");
                }
            else {
                *tcp1 = '\0';
                strcpy(rootname, curname);
                *tcp1 = '.';
                }
        
            strcpy(filename, curarea);
            strcat(filename, curname);
        
            if ((dvifp=BINARYOPEN(filename,"r")) == NULL)  {
        	fprintf(stderr,"\n");
        	fprintf(stderr,"%s: can't find DVI file \"%s\"\n\n", G_progname, filename);
        	AbortRun(1);
                }
        
            strcpy(G_Logname, curname);
            strcat(G_Logname, ".log");
	    }
	argind++;
        }

    if (dvifp == NULL)  {
	fprintf(stderr, 
                "\nusage: %s [-a area] [-c] [-h] [-o option] [-p] [-s] [-r] [-f n] [-t n] [-m{0|h|1|2|3|4|  mag] [-a fontarea] dvifile\n\n", 
                G_progname);
	AbortRun(1);
        }
    PXLpathlen = SplitPath(PXLpath);
    TFMpathlen = SplitPath(TFMpath);
}


/*-->SplitPath*/
/*********************************************************************/
/********************************  SplitPath  ************************/
/*********************************************************************/

int
SplitPath(path)
char **path;
{
    char *area, *tmp;
    int pathlen = 0;
    strcpy ( area = malloc(strlen(*path)+1), *path);
    do {
	path[pathlen++] = area;
	tmp = index (area, ':');
	if (tmp) {
	    *tmp = '\0';
	    area = tmp+1;
	}
    } while (tmp && *area);
    return pathlen;
}

/*-->DoConv*/
/*********************************************************************/
/********************************  DoConv  ***************************/
/*********************************************************************/

int DoConv(num, den, convResolution)
{
    register float conv;
    conv = ((float)num/(float)den) * 
#ifdef USEGLOBALMAG
/*	ActualFactor(mag) * why was this in as Actual Factor?  jls */
	((float) mag/1000.0) *
#endif
	((float)convResolution/254000.0);
    return((int) (1.0 / conv + 0.5));
}


/*-->DoSpecial*/
/*********************************************************************/
/*****************************  DoSpecial  ***************************/
/*********************************************************************/

typedef enum {None, String, Integer, Number, Dimension} ValTyp;

typedef struct {
        char    *Key;           /* the keyword string */
        char    *Val;           /* the value string */
        ValTyp  vt;             /* the value type */
        union {                 /* the decoded value */
            int  i;
            float n;
            } v;
        } KeyWord;

typedef struct {
        char    *Entry;
        ValTyp  Type;
        } KeyDesc;

#define PSFILE 0
#define PSTEXT 7
#define NOSAVE 8
KeyDesc KeyTab[] = {{"psfile", String},
                    {"hsize", Dimension},
                    {"vsize", Dimension},
                    {"hoffset", Dimension},
                    {"voffset", Dimension},
                    {"hscale", Number},
                    {"vscale", Number},
		    {"pstext", String},
	      	    {"nosave", None}};

#define NKEYS (sizeof(KeyTab)/sizeof(KeyTab[0]))

void
DoSpecial( str, n )          /* interpret a \special command, made up of keyword=value pairs */
char    *str;
int n;
{ 
        char spbuf[STRSIZE]; 
	BOOLEAN in_files = FALSE, tosave = TRUE, begun = FALSE;
        KeyWord k;
        int i;

        str[n] = '\0';
        spbuf[0] = '\0';

        SetPosn(hh, vv);

	while( (str=GetKeyStr(str,&k)) != NULL ) {      /* get all keyword-value pairs */
		/* for compatibility, single words are taken as file names */
		if( k.vt == None && access(k.Key,0) == 0) {
			strcpy(spbuf, k.Key);
			if (!begun) {
				if (tosave)
				  EMITS("@beginspecial\n");
				begun = TRUE;
			}
			if (!in_files) {
				if (tosave)
				  EMITS("@setspecial\n");
				in_files = TRUE;
			}
			CopyFile(spbuf);
		} else if( GetKeyVal( &k, KeyTab, NKEYS, &i ) && i != -1 ) {
			if (i == NOSAVE) {
				tosave = FALSE;
			} else {
				if (!begun) {
					if (tosave)
					  EMITS("@beginspecial\n");
					begun = TRUE;
				}
				if( i == PSFILE ) {
					strcpy(spbuf, k.Val);
					if (!in_files) {
						if (tosave)
						  EMITS("@setspecial\n");
						in_files = TRUE;
					}
					CopyFile(spbuf);
				} else if(i == PSTEXT) {
					EMITS(k.Val);
					EMITC('\n');
				} else {
					/* the keywords are simply output as PS procedure calls */
					if (in_files) {
						Warning("  Files must appear after other commands in \\special - \"%s\" ignored", k.Key);
					} else {
						EMIT(outfp, "%f @%s\n", k.v.n, KeyTab[i].Entry);
					}
				}
			}
		} else
		  Warning("  Invalid keyword or value in \\special - \"%s\" ignored", k.Key );
	}

	if (tosave)
	  EMITS("@endspecial\n");
}

/*-->EmitBitmap*/
/**********************************************************************/
/****************************  EmitBitmap  ****************************/
/**********************************************************************/

void
EmitBitmap(c, ce, command, global)              /* output a character */
int c;
char * command;
struct char_entry *ce;
int global;			/* write to prolog if true and prolog != NULL */
{
		/* Output in PostScript coord system (y +ive up, x +ive right)
		   (0,0) of char bitmap at lower left.  Output scan lines
                           from bottom to top */
        int i;
        register int j;
        register unsigned char *sl;
        register int cc;
        int nbpl;

	if (global)
	  GEMITS("[<");
	else
	  EMITS("[<");
        cc = 2;
        nbpl = (ce->width + 7) >> 3;
        for(i = ce->height-1;  i >= 0;  i--) {
                sl = (unsigned char *)(ce->where.address.pixptr + i*nbpl);
                for(j = 0;  j < nbpl;  j++, sl++) {
                        if( cc > 100 ) {
                                if (global)
				  GEMITS("\n  ");
				else
				  EMITS("\n  ");
				cc = 2; }
			if (global)
			  GEMITH(*sl);
			else
			  EMITH(*sl);
                        cc += 2;
                        }
                }
	if (global)
	  GEMIT(OUTFP, "> %d %d %d %d %d] %d %s\n", 
		ce->width, ce->height, ce->xOffset,
		(((int)ce->height)-ce->yOffset)-1, ce->dx, c, command);
	else
	  EMIT(outfp, "> %d %d %d %d %d] %d %s\n", 
		ce->width, ce->height, ce->xOffset,
		(((int)ce->height)-ce->yOffset)-1, ce->dx, c, command);

#ifdef STATS
        Snbpxl += nbpl*ce->height;
        fontptr->nbpxl += nbpl*ce->height;
        Sonbpx += (ce->width*ce->height + 7) >> 3;
        Sndc ++;
#endif
}

/*-->EmitChar*/
/**********************************************************************/
/****************************  EmitChar  ******************************/
/**********************************************************************/

void
EmitChar(c, ce)              /* output a character */
int c;
struct char_entry *ce;
{

#ifdef BUDGET
	/* don't load character permanently if it's too big */
	if (BIGCHAR(ce)) return;
#endif
	if( fontptr->ncdl == 0 ) {    /* open font dict before first char */
#ifdef BUDGET
	    if (fontcost(fontptr) >= budget) return;
	    budget -= fontcost(fontptr);
#endif
	    GEMIT(OUTFP,"/%s @newfont\n", fontptr->psname );
	}
 
	if( fontptr != prevfont ) {   /* because this isn't done on pass 0 */
		if (prolog)
		  GEMIT(OUTFP,"%s @sf\n", fontptr->psname);
		EMIT(outfp,"%s @sf\n", fontptr->psname);
		prevfont = fontptr;
		}

	EmitBitmap(c, ce, "@dc", TRUE);
        fontptr->ncdl ++;

	ce->where.isloaded = PERMANENT;
#ifdef BUDGET
	budget -= charcost(ce);
#endif

}


/*-->Fatal*/
/**********************************************************************/
/******************************  Fatal  *******************************/
/**********************************************************************/

void
Fatal(fmt, a, b, c)/* issue a fatal error message */
char *fmt;	/* format */
char *a, *b, *c;	/* arguments */

{
    if (G_logging == 1 && G_logfile)
    {
	fprintf(G_logfp, "%s: FATAL--", G_progname);
	fprintf(G_logfp, fmt, a, b, c);
	fprintf(G_logfp, "\n");
    }

    fprintf(stderr,"\n");
    fprintf(stderr, "%s: FATAL--", G_progname);
    fprintf(stderr, fmt, a, b, c);
    fprintf(stderr, "\n\n");
    if (G_logging == 1) printf("Log file created\n");
#ifdef CREOPT
    if (G_create && outfp != NULL) {
        fclose(outfp);
        unlink(outfname);
        }
#endif CREOPT
    AbortRun(1);
}


/*-->FindPath*/
/**********************************************************************/
/****************************  FindPath  ******************************/
/**********************************************************************/

int
FindPath (path, len, file, out)
char **path, *file, *out;
int len;
{
	int i;
	char fn[STRSIZE];
	for (i=0; i<len; i++) {
	  sprintf (fn, "%s/%s", path[i], file);
	  if (! access (fn, 4)) {
	    strcpy( out, fn );
	    return i;
	  }
	}
	return -1;
}


/*-->FindPostAmblePtr*/
/**********************************************************************/
/************************  FindPostAmblePtr  **************************/
/**********************************************************************/

void
FindPostAmblePtr(postambleptr)
long	*postambleptr;

/* this routine will move to the end of the file and find the start
    of the postamble */

{
    int     i;

    fseek (dvifp, (long) 0, 2);   /* goto end of file */
    *postambleptr = ftell (dvifp) - 4;
    fseek (dvifp, *postambleptr, 0);

    while (TRUE) {
	fseek (dvifp, --(*postambleptr), 0);
	if (((i = NoSignExtend(dvifp, 1)) != 223) &&
	    (i != DVIFORMAT))
	    Fatal ("Bad end of DVI file");
	if (i == DVIFORMAT)
	    break;
    }
    fseek (dvifp, (*postambleptr) - 4, 0);
    (*postambleptr) = NoSignExtend(dvifp, 4);
    fseek (dvifp, *postambleptr, 0);
}


/*-->GetBytes*/
/**********************************************************************/
/*****************************  GetBytes  *****************************/
/**********************************************************************/

void
GetBytes(fp, cp, n)	/* get n bytes from file fp */
register FILE *fp;	/* file pointer	 */
register char *cp;	/* character pointer */
register int n;		/* number of bytes  */

{
    while (n--)
	*cp++ = getc(fp);
}


/*-->GetFontDef*/
/**********************************************************************/
/**************************** GetFontDef  *****************************/
/**********************************************************************/

void
GetFontDef()

/***********************************************************************
   Read the font  definitions as they  are in the  postamble of the  DVI
   file.
***********************************************************************/

{
    char    str[50], *calloc ();
    unsigned char   byte;
    int     i, fnamelen;

    while (((byte = NoSignExtend(dvifp, 1)) >= FNT_DEF1) &&
	(byte <= FNT_DEF4))
	    ReadFontDef (NoSignExtend(dvifp, byte-FNT_DEF1+1));
    if (byte != POST_POST)
	Fatal ("POST_POST missing after fontdefs");
}


/*-->GetKeyStr*/
/**********************************************************************/
/*****************************  GetKeyStr  ****************************/
/**********************************************************************/

        /* extract first keyword-value pair from string (value part may be null)
         * return pointer to remainder of string
         * return NULL if none found
         */

char    KeyStr[STRSIZE];
char    ValStr[STRSIZE];

char *GetKeyStr( str, kw )
char    *str;
KeyWord *kw;
{
        char *s, *k, *v, t;

        if( !str ) return( NULL );

        for( s=str; *s == ' '; s++ ) ;                  /* skip over blanks */
        if( *s == '\0' ) return( NULL );

        for( k=KeyStr;                          /* extract keyword portion */
             *s != ' ' && *s != '\0' && *s != '='; 
             *k++ = *s++ ) ;
        *k = '\0';
        kw->Key = KeyStr;
        kw->Val = v = NULL;
        kw->vt = None;

        for( ; *s == ' '; s++ ) ;                       /* skip over blanks */
        if( *s != '=' )                         /* look for "=" */
                return( s );

        for( s++ ; *s == ' '; s++ ) ;                   /* skip over blanks */
        if( *s == '\'' || *s == '\"' )          /* get string delimiter */
                t = *s++;
        else
                t = ' ';
        for( v=ValStr;                          /* copy value portion up to delim */
             *s != t && *s != '\0';
             *v++ = *s++ ) ;
        if( t != ' ' && *s == t ) s++;
        *v = '\0';
        kw->Val = ValStr;
        kw->vt = String;

        return( s );
}


/*-->GetKeyVal*/
/**********************************************************************/
/*****************************  GetKeyVal  ****************************/
/**********************************************************************/

        /* get next keyword-value pair
         * decode value according to table entry
         */

int GetKeyVal( kw, tab, nt, tno)
KeyWord *kw; 
KeyDesc tab[];
int     nt;
int     *tno;
{
        int i;
        char c = '\0';

        *tno = -1;

        for(i=0; i<nt; i++)
                if( IsSame(kw->Key, tab[i].Entry) ) {
                        *tno = i;
                        switch( tab[i].Type ) {
                                case None: 
                                        if( kw->vt != None ) return( FALSE );
                                        break;
                                case String:
                                        if( kw->vt != String ) return( FALSE );
                                        break;
                                case Integer:
                                        if( kw->vt != String ) return( FALSE );
                                        if( sscanf(kw->Val,"%d%c", &(kw->v.i), &c) != 1
                                            || c != '\0' ) return( FALSE );
                                        break;
                                case Number:
                                case Dimension:
                                        if( kw->vt != String ) return( FALSE );
                                        if( sscanf(kw->Val,"%f%c", &(kw->v.n), &c) != 1
                                            || c != '\0' ) return( FALSE );
                                        break;
                                }
                        kw->vt = tab[i].Type;
                        return( TRUE );
                        }

        return( TRUE );
}


/*-->HasBeenRead*/
/**********************************************************************/
/***************************  HasBeenRead  ****************************/
/**********************************************************************/

int
HasBeenRead(k)
int k;
{
    struct font_entry *ptr;

    ptr = hfontptr;
    while ((ptr!=NULL) && (ptr->k!=k))
	ptr = ptr->next;
    return( ptr != NULL );
}


/*-->InPageList*/
/**********************************************************************/
/******************************  InPageList  **************************/
/**********************************************************************/
/* Return true iff i is one of the desired output pages */

int InPageList(i)
int i;{
    register struct pagelist   *pl = PageList;

    while (pl) {
	    if ( i >= pl -> ps_low && i <= pl -> ps_high)
		return 1;		/* success */
	pl = pl -> pl_alt;
    }
    return 0;
}


#ifdef BUILTIN
/*-->IsBuiltin*/
/**********************************************************************/
/*******************************  IsBuiltin  **************************/
/**********************************************************************/
int IsBuiltIn(a)
char *a;
{
	struct BuiltIn *c;
	for(c=BuiltIns; c; c=c->next)
		if(strcmp(c->TeXname,a)==0) return(1);
	return(0);
}
#endif


/*-->IsSame*/
/**********************************************************************/
/*******************************  IsSame  *****************************/
/**********************************************************************/

int IsSame(a, b)        /* compare strings, ignore case */
char *a, *b;
{
        for( ; *a != '\0'; )
                if( tolower(*a++) != tolower(*b++) ) return( FALSE );
        return( *a == *b ? TRUE : FALSE );
}


/*-->lcase*/
/**********************************************************************/
/****************************  lcase  *********************************/
/**********************************************************************/

void lcase(s)
char *s;
{
        char *t;
        for(t=s; *t != '\0'; t++)
	  *t = tolower(*t);
        return;
}


/*-->MoveDown*/
/**********************************************************************/
/****************************  MoveDown  ******************************/
/**********************************************************************/

void
MoveDown(a)
int a;
{
    int vtemp;
    vtemp = PixRound (v += a, vconv);
    if (fontptr && abs(a)<5*fontptr->font_space) {
	vv += PixRound(a, vconv);
	if (vtemp>vv+max_drift) vv = vtemp-max_drift;
	else if (vtemp<vv-max_drift) vv = vtemp+max_drift;
    }
    else vv = vtemp;
}


/*-->MoveOver*/
/**********************************************************************/
/****************************  MoveOver  ******************************/
/**********************************************************************/

void
MoveOver(b)
int b;
{
    int htemp;
    htemp = PixRound (h += b, hconv);
    if (fontptr && b<fontptr->font_space && b>-4*fontptr->font_space) {
	hh += PixRound(b, hconv);
	if (htemp>hh+max_drift) hh = htemp-max_drift;
	else if (htemp<hh-max_drift) hh = htemp+max_drift;
    }
    else hh = htemp;
}

/*-->MoveRight*/
/**********************************************************************/
/****************************  MoveRight  ******************************/
/**********************************************************************/

void
MoveRight(b)
int b;
{
    int htemp;
    htemp = PixRound (h += b, hconv);
    if (htemp>hh+max_drift) hh = htemp-max_drift;
    else if (htemp<hh-max_drift) hh = htemp+max_drift;
}


/*-->NoSignExtend*/
/**********************************************************************/
/***************************  NoSignExtend  ***************************/
/**********************************************************************/

int
NoSignExtend(fp, n)	/* return n byte quantity from file fd */
register FILE *fp;	/* file pointer    */
register int n;		/* number of bytes */

{
    register int x;	/* number being constructed */

    x = 0;
    while (n--)  {
	x <<= 8;
	x |= getc(fp);
    }
    return(x);
}


/*-->OpenFontFile*/
/**********************************************************************/
/************************** OpenFontFile  *****************************/
/**********************************************************************/

void
OpenFontFile()
/***********************************************************************
    The original version of this dvi driver reopened the font file  each
    time the font changed, resulting in an enormous number of relatively
    expensive file  openings.   This version  keeps  a cache  of  up  to
    MAXOPEN open files,  so that when  a font change  is made, the  file
    pointer, pxlfp, can  usually be  updated from the  cache.  When  the
    file is not found in  the cache, it must  be opened.  In this  case,
    the next empty slot  in the cache  is assigned, or  if the cache  is
    full, the least used font file is closed and its slot reassigned for
    the new file.  Identification of the least used file is based on the
    counts of the number  of times each file  has been "opened" by  this
    routine.  On return, the file pointer is always repositioned to  the
    beginning of the file.

***********************************************************************/
{
    register int i,least_used,current;

#ifdef DEBUG
    if (Debug) printf("Open Font file\n");
#endif
    if (pfontptr == fontptr)
        return;                 /* we need not have been called */

    for (current = 1;
	(current <= nopen) &&
	    (pixel_files[current].pixel_file_id != fontptr->font_file_id);
	++current)
	;                       /* try to find file in open list */

    if (current <= nopen)       /* file already open */
    {
	if( (pxlfp = pixel_files[current].pixel_file_id) != NO_FILE )
	        fseek(pxlfp,0,0);	/* reposition to start of file */
    }
    else                        /* file not in open list */
    {
        if (nopen < MAXOPEN)    /* just add it to list */
            current = ++nopen;
	else                    /* list full -- find least used file, */
	{                       /* close it, and reuse slot for new file */
	    least_used = 1;
            for (i = 2; i <= MAXOPEN; ++i)
	        if (pixel_files[least_used].use_count >
                    pixel_files[i].use_count)
		    least_used = i;
            if (pixel_files[least_used].pixel_file_id != NO_FILE) {
                FILE *fid;
                struct font_entry *fp;

                fid = pixel_files[least_used].pixel_file_id;
                fp=hfontptr;    /* mark file as being closed in the entry */
                while (fp!=NULL && fp->font_file_id != fid) 
                        fp = fp->next;
                if (fp==NULL)
		  Fatal("Open file %x not found in font entry list.\n", fid);
                else {
                        fp->font_file_id = NULL;
#ifdef STATS
                        if (Stats)
                                fprintf(stderr, "PXL file %s closed.\n", fp->name);
#endif
                        }
 	        fclose( fid );
                }
	    current = least_used;
        }
        if ((pxlfp=BINARYOPEN(fontptr->name,"r")) == NULL) {
	    Warning("PXL file %s could not be opened",fontptr->name);
            pxlfp = NO_FILE;
            }
        else {
#ifdef STATS
            if (Stats) 
                fprintf(stderr, "PXL file %s opened.\n", fontptr->name);
#endif
            }
	pixel_files[current].pixel_file_id = pxlfp;
	pixel_files[current].use_count = 0;
    }
    pfontptr = fontptr;			/* make previous = current font */
    fontptr->font_file_id = pxlfp;	/* set file identifier */
    pixel_files[current].use_count++;	/* update reference count */
}

#ifdef CREOPT
/*-->OpenOutput*/   /* generate a unique file name and open it */
/**********************************************************************/
/*************************** OpenOutput *******************************/
/**********************************************************************/


FILE*
OpenOutput()
{
        FILE*   fp;
        long t;
        int  n = 0;
        char *p, *pp, b[256];
        int nd;

        time( &t );
        t = t % 100000;
        strcpy( outfname, SPOOLFILE );
        sprintf( b, "%s.%s.%x", logname(), rootname, t );
        if( (nd=strlen(b)-MAXFLEN) > 0 ) {
               for(pp=(p=rindex(b,'.')); p && *p != '\0'; *(pp-nd) = *p++, pp++) ;
               *(pp-nd) = '\0';
               }
        strcat( outfname, b );

        while( access(outfname,0) == 0 ) {
                n ++;
                if( n > 10 ) 
                        Fatal( "Unable to create a unique output file name: %s\n", outfname );
                strcpy( outfname, SPOOLFILE );
                sprintf( b, "%s.%s.%x.%d", logname(), rootname, t, n );
                if( (nd=strlen(b)-MAXFLEN) > 0 ) {
                        for(pp=(p=rindex(b,'.')); p && *p != '\0'; *(pp-nd) = *p++, pp++) ;
                        *(pp-nd) = '\0';
                        }
                strcat( outfname, b );
                }

        if( (fp=fopen(outfname,"w")) == NULL )
                Fatal("Unable to create output file: %s\n", outfname);

        return( fp );
}
#endif CREOPT

struct pagelist *
InstallPL (pslow, pshigh)
int pslow, pshigh; {
    register struct pagelist   *pl;

    pl = (struct pagelist *) malloc (sizeof *pl);
    if (pl == 0)
	Fatal ("Not enough memory");
    pl -> pl_alt = PageList;
    pl -> ps_low = pslow;
    pl -> ps_high = pshigh;
    PageList = pl;
}

/* Parse a string representing a list of pages.  Return 0 iff ok.  As a
   side effect, the page selection(s) is (are) prepended to PageList. */

ParsePages (s)
register char  *s; {
    register int    c,		/* current character */
                    n,		/* current numeric value */
		    innumber;	/* true => gathering a number */
    int     ps_low, ps_high,
            range,		/* true => saw a range indicator */
	    negative;		/* true => number being built is negative */

#define white(x) ((x) == ' ' || (x) == '\t' || (x) == ',')

    range = 0;
    innumber = 0;
    for (;;) {
	c = *s++;
	if ( !innumber && !range) {/* nothing special going on */
	    if (c == 0)
		return 0;
	    if (white (c))
		continue;
	}
	if (c == '-' && !innumber) {
		innumber++;
		negative++;
		n = 0;
		continue;
	}
	if (isdigit (c)) {	/* accumulate numeric value */
	    if (!innumber) {
		innumber++;
		negative = 0;
		n = c - '0';
		continue;
	    }
	    n *= 10;
	    n += negative ? '0' - c : c - '0';
	    continue;
	}
	if (c == '-' || c == ':') {/* here's a range */
	    if (range)
		return (-1);
	    if (innumber) {	/* have a lower bound */
		ps_low = n;
	    }
	    else
		ps_low = -1000000;
	    range++;
	    innumber = 0;
	    continue;
	}
	if (c == 0 || white (c)) {/* end of this range */
	    if (!innumber) {	/* no upper bound */
		ps_high = 1000000;
		if (!range)	/* no lower bound either */
		    ps_low = -1000000;
	    }
	    else {		/* have an upper bound */
		ps_high = n;
		if (!range) {	/* no range => lower bound == upper */
		    ps_low = ps_high;
		}
	    }

	    InstallPL (ps_low, ps_high);

	    if (c == 0)
		return 0;
	    range = 0;
	    innumber = 0;
	    continue;
	}
	return (-1);
    }
#undef white
}
	

/*-->PixRound*/
/**********************************************************************/
/*****************************  PixRound  *****************************/
/**********************************************************************/

int
PixRound(x, conv)	/* return rounded number of pixels */
register int x;		/* in DVI units     */
int conv;		/* conversion factor */
{
    return((int)((x + (conv >> 1)) / conv));
}


/*-->RuleRound*/
/**********************************************************************/
/*****************************  RuleRound  ****************************/
/**********************************************************************/

int
RuleRound(x, conv)	/* return rounded number of pixels */
register int x;		/* in DVI units     */
int conv;		/* conversion factor */
{
    return((int)((x + conv - 1) / conv));
}


#ifdef BUILTIN
/*-->PSname*/
/**********************************************************************/
/*****************************  PSname  *******************************/
/**********************************************************************/
char *PSname(a)
char *a;
{
	struct BuiltIn *c;
	for(c=BuiltIns; c; c=c->next)
		if(strcmp(c->TeXname,a)==0) return( c->Postscript );
	return("<NULLFONT>");
}
#endif


/*-->PutInt*/
/**********************************************************************/
/*****************************  PutInt  *******************************/
/**********************************************************************/

void
PutInt(n)               /* output an integer followed by a space */
register int n;
{
    char buf[10];
    register char *b;

    if( n == 0 )
        EMITC('0'); 
    else {
        if( n < 0 ) {
            EMITC('-');
            n = -n;
            }
    
        for(b=buf;  n>0;  ) {
            *b++ = digit[n%10];
            n /= 10;
            }
    
        for( ; b>buf; )
            EMITC(*--b);
        }

    EMITC(' ');
}


/*-->PutOct*/
/**********************************************************************/
/*****************************  PutOct  *******************************/
/**********************************************************************/

void
PutOct(n)               /* output an 3 digit octal number preceded by a "\" */
register int n;
{                  
    EMITC( '\\' ); 
    EMITC( digit[(n&0300)>>6] );
    EMITC( digit[(n&0070)>>3] ); 
    EMITC( digit[n&0007] );
}

#ifdef PK

/*-->skip_specials*/
/**********************************************************************/
/****************************  skip_specials  *************************/
/**********************************************************************/

void
skip_specials()
{
    int i;
    do {
	switch (input_byte = getc(pxlfp)) {
	    case PKXXX1:
	    case PKXXX2:
	    case PKXXX3:
	    case PKXXX4:
			    i =  NoSignExtend(pxlfp, input_byte-PKXXX1+1);
			    while (--i>=0) getc(pxlfp);
			    break;
	    case PKYYY:	i = NoSignExtend(pxlfp, 4);
			    break;
	    case PKPOST:	return;
	    }
	} while (input_byte>=PKXXX1);
}

#endif /* PK */

/*-->ReadFontDef*/
/**********************************************************************/
/****************************  ReadFontDef  ***************************/
/**********************************************************************/

int
ReadFontDef(k)
int k;
{
    int t, i;
#ifdef PK
    int pl, cc;
#endif /* PK */
    register struct font_entry *tfontptr;/* temporary font_entry pointer   */
    register struct char_entry *tcharptr;/* temporary char_entry pointer  */
    char *direct, *tcp, *tcp1;
    int found;
    char curarea[STRSIZE];
    int cmag;
    int nmag;
    char nname[128];
    double z;

#define TfmCalc4(fp) ((int)((double)SignExtend(fp,4) * z))
#define TfmCalc3(fp) ((int)((double)SignExtend(fp,3) * z))

    if ((tfontptr = NEW(struct font_entry)) == NULL)
	Fatal("can't malloc space for font_entry");
    tfontptr->next = hfontptr;
    tfontptr->font_file_id = NULL;
    fontptr = hfontptr = tfontptr;
    for (i = FIRSTPXLCHAR; i <= LASTPXLCHAR; i++) {
	tcharptr = &(tfontptr->ch[i]);
	tcharptr->width = 0;
	tcharptr->height = 0;
	tcharptr->xOffset= 0;
	tcharptr->yOffset = 0;
	tcharptr->where.isloaded = NOTLOADED;
	tcharptr->where.address.fileOffset = NONEXISTANT;
	tcharptr->tfmw = 0;
	tcharptr->dx = 0;
	}
    tfontptr->ncdl = 0;
#ifdef STATS
    tfontptr->nbpxl = 0;
    tfontptr->ncts = 0;
#endif

    tfontptr->k = k;
    tfontptr->c = NoSignExtend(dvifp, 4); /* checksum */
    tfontptr->s = NoSignExtend(dvifp, 4); /* scale factor */
    tfontptr->d = NoSignExtend(dvifp, 4); /* design size */
    tfontptr->a = NoSignExtend(dvifp, 1); /* area length for font name */
    tfontptr->l = NoSignExtend(dvifp, 1); /* device length */
    GetBytes(dvifp, tfontptr->area, tfontptr->a);
    tfontptr->area[tfontptr->a] = '\0';
    GetBytes(dvifp, tfontptr->n, tfontptr->l);
    tfontptr->n[tfontptr->l] = '\0';
    tfontptr->font_space = tfontptr->s/6;
    tfontptr->font_mag = (int)((ActualFactor((int)(((float)tfontptr->s/
    			(float)tfontptr->d)*1000.0 + 0.5)) * 
#ifdef USEGLOBALMAG
			ActualFactor(mag) *
#endif
#ifdef PK
			(float)resolution
#else
  			(float)resolution * 5.0
#endif /* PK */
                           ) + 0.5);
    z = (double)tfontptr->s / (double)(1<<20);

#ifdef BUILTIN
    if( IsBuiltIn(tfontptr->n) ){
	static char ptfmname[100] = "";
	static FILE* tfmfl = NULL;
	char tfmname[100];
	int lh,bc,ec,i,nw,tfmwidth[NPXLCHARS],v;
        sprintf(tfontptr->psname, "%s.%d", tfontptr->n, tfontptr->font_mag);
	/*** Read in TFM widths from TFM file (not PXL file) ***/
	if (tfontptr->a) sprintf(tfmname, "%s/%s.tfm", tfontptr->area,
				 tfontptr->n);
	else {
	  sprintf(tfmname, "%s.tfm", tfontptr->n);
	  FindPath (TFMpath, TFMpathlen, tfmname, tfmname);
	}
	if (strcmp (tfmname, ptfmname)) {
	  if (tfmfl != NULL) fclose (tfmfl);
	  if((tfmfl=fopen(tfmname,"r"))==NULL){
		tfontptr->builtin = 0;
		tfontptr->font_file_id = NO_FILE;
		*ptfmname = '\0';
		fprintf(stderr,"Can't find LaserWriter TFM file %s\n",
			tfmname);
		return;
		}
	  strcpy (ptfmname, tfmname);
	}
	tfontptr->builtin = 1;
	fseek(tfmfl,2L,0);
	lh = NoSignExtend (tfmfl, 2);
	bc = NoSignExtend (tfmfl, 2);
	ec = NoSignExtend (tfmfl, 2);
	nw = NoSignExtend (tfmfl, 2);
	fseek(tfmfl,(long)(24+(lh+ec-bc+1)*4),0);
	for (i=0; i<nw; i++)
          tfmwidth[i] = TfmCalc4 (tfmfl);
	fseek(tfmfl,(long)lh*4+24,0);
	for(i=bc; i<=ec; i++) {
	  v=tfmwidth[NoSignExtend(tfmfl,1)];
	  tcharptr = &(tfontptr->ch[i]);
	  tcharptr->tfmw = v;
	  tcharptr->dx = PixRound (v, hconv);
	  getc(tfmfl);getc(tfmfl);getc(tfmfl);
	}
	return;
      }

    tfontptr->builtin=0;
#endif

    if (!findfile(PXLpath,PXLpathlen,
		       tfontptr->area,tfontptr->n,tfontptr->font_mag,
		       tfontptr->name, nname, &nmag))
     Fatal("no font %s.%d",tfontptr->n,tfontptr->font_mag);

    sprintf(tfontptr->psname, "%s.%d", tfontptr->n, tfontptr->font_mag);
    if (tfontptr != pfontptr)
	OpenFontFile();
    if ( pxlfp == NO_FILE ) {                /* allow missing pxl files */
        tfontptr->magnification = 0;
        tfontptr->designsize = 0;
        return;
        }

#ifndef PK
    if ((t = NoSignExtend(pxlfp, 4)) != PXLID)
	Fatal("PXL ID = %d, can only process PXL ID = %d files",
	      t, PXLID);
    fseek(pxlfp, -20, 2);
#else
    if ((NoSignExtend(pxlfp, 1) != PKPRE)||((t = NoSignExtend(pxlfp, 1)) != PKID))
	Fatal("PK ID = %d, can only process PK ID = %d files",
	      t, PKID);
    i = NoSignExtend(pxlfp, 1);
    for (;i>0;i--) getc(pxlfp);
    tfontptr->designsize = NoSignExtend(pxlfp, 4);
#endif /* PK */
    t = NoSignExtend(pxlfp, 4);
    if ((tfontptr->c != 0) && (t != 0) && (tfontptr->c != t))
	Warning("font = \"%s\",\n-->font checksum = %d,\n-->dvi checksum = %d",
	        tfontptr->name, t, tfontptr->c);
#ifndef PK
    tfontptr->magnification = NoSignExtend(pxlfp, 4);
    tfontptr->designsize = NoSignExtend(pxlfp, 4);
    fseek(pxlfp, NoSignExtend(pxlfp, 4) * 4, 0);

    for (i = FIRSTPXLCHAR; i <= LASTPXLCHAR; i++) {
	tcharptr = &(tfontptr->ch[i]);
	tcharptr->width = NoSignExtend(pxlfp, 2);
	tcharptr->height = NoSignExtend(pxlfp, 2);
	tcharptr->xOffset= SignExtend(pxlfp, 2);
	tcharptr->yOffset = SignExtend(pxlfp, 2);
	tcharptr->where.address.fileOffset = NoSignExtend(pxlfp, 4) * 4;
	tcharptr->tfmw = TfmCalc4(pxlfp);
	tcharptr->dx = PixRound(tcharptr->tfmw, hconv);
    }
#else
    t = NoSignExtend(pxlfp, 4); /* hppp */
    tfontptr->magnification = (int) (t * 72.27 *5 / 65536 + 0.5);
    if (t != NoSignExtend(pxlfp, 4)) /* vppp */
	Warning("font = \"%s\",\n-->font hppp != vppp",
	        tfontptr->name);
    while  (skip_specials(), input_byte != PKPOST) {
	
	dyn_f = input_byte >> 4;
	t = (input_byte >> 3)&1;
	input_byte &= 7;
	if (input_byte == 7) {
	    pl = NoSignExtend(pxlfp, 4);
	    cc = NoSignExtend(pxlfp, 4) & 0177;
	    tcharptr = &(tfontptr->ch[cc]);
	    tcharptr->tfmw = TfmCalc4(pxlfp);
	    tcharptr->dx = NoSignExtend(pxlfp, 4)>>16;
	    i = NoSignExtend(pxlfp, 4);
	    tcharptr->width = NoSignExtend(pxlfp, 4);
	    tcharptr->height = NoSignExtend(pxlfp, 4);
	    tcharptr->xOffset= SignExtend(pxlfp, 4);
	    tcharptr->yOffset = SignExtend(pxlfp, 4);
	    pl -= 28;
	}
	else if (input_byte > 3) {
	    pl = NoSignExtend(pxlfp, 2) + ((input_byte-4)<<16);
	    cc = NoSignExtend(pxlfp, 1) & 0177;
	    tcharptr = &(tfontptr->ch[cc]);
	    tcharptr->tfmw = TfmCalc3(pxlfp);
	    tcharptr->dx = NoSignExtend(pxlfp, 2);
	    tcharptr->width = NoSignExtend(pxlfp, 2);
	    tcharptr->height = NoSignExtend(pxlfp, 2);
	    tcharptr->xOffset= SignExtend(pxlfp, 2);
	    tcharptr->yOffset = SignExtend(pxlfp, 2);
	    pl -= 13;
	}
	else {
	    pl = NoSignExtend(pxlfp, 1) + ((input_byte)<<8);
	    cc = NoSignExtend(pxlfp, 1) & 0177;
	    tcharptr = &(tfontptr->ch[cc]);
	    tcharptr->tfmw = TfmCalc3(pxlfp);
	    tcharptr->dx = NoSignExtend(pxlfp, 1);
	    tcharptr->width = NoSignExtend(pxlfp, 1);
	    tcharptr->height = NoSignExtend(pxlfp, 1);
	    tcharptr->xOffset= SignExtend(pxlfp, 1);
	    tcharptr->yOffset = SignExtend(pxlfp, 1);
	    pl -= 8;
	}
	tcharptr->where.address.fileOffset = ftell(pxlfp);
	tcharptr->turn_on = t;
	tcharptr->dyn_f = dyn_f;
	fseek(pxlfp,pl,1);
    }
#endif /* PK */
}


#ifdef BUILTIN
/*-->ReadMap*/
/**********************************************************************/
/****************************  ReadMap  *******************************/
/**********************************************************************/

ReadMap()
{
	FILE *map;
	char TeXn[STRSIZE], PSn[STRSIZE], filen[STRSIZE];
	struct BuiltIn *p=0, *q;
	int c, status;
	int pathnr = 0;

	while ( (c = FindPath (&TFMpath[pathnr], TFMpathlen-pathnr, FONTMAP, filen)) >= 0) {
	    pathnr += c+1;
	    if((map=fopen(filen,"r"))==NULL)
		    Fatal("Can't open TeX->PostScript font map %s\n",filen);
	    c = 1;
	    while((status=fscanf(map,"%s%s", TeXn, PSn))==2) {
		q = (struct BuiltIn *) malloc(sizeof (struct BuiltIn));
		q->next = 0;
		strcpy (q->TeXname, TeXn);
		strcpy (q->Postscript, PSn);
		if (p) p->next = q;
		else BuiltIns = q;
		p = q;
		c++;
	      }
	    if(status!=EOF)
		    Fatal("Syntax error in %s, line %d\n", filen, c);
	    fclose(map);
	}
}
#endif


/*-->ReadPostAmble*/
/**********************************************************************/
/**************************  ReadPostAmble  ***************************/
/**********************************************************************/

void
ReadPostAmble(load)
int     load;
/***********************************************************************
    This  routine  is  used  to  read  in  the  postamble  values.    It
    initializes the magnification and checks  the stack height prior  to
    starting printing the document.
***********************************************************************/
{
    FindPostAmblePtr (&postambleptr);
    if (NoSignExtend(dvifp, 1) != POST)
	Fatal ("POST missing at head of postamble");
#ifdef DEBUG
    if (Debug) fprintf (stderr, "got POST command\n");
#endif
    ppagep = NoSignExtend(dvifp, 4);
    num = NoSignExtend(dvifp, 4);
    den = NoSignExtend(dvifp, 4);
    mag = NoSignExtend(dvifp, 4);
#ifdef USEGLOBALMAG
    if( usermag > 0 && usermag != mag )
        fprintf(stderr, "DVI magnification of %d over-ridden by user mag of %d\n", mag, usermag );
#endif
    if( usermag > 0 ) mag = usermag;
#ifndef USEGLOBALMAG
    if( mag != 1000 ) fprintf(stderr, "Magnification of %d ignored.\n", mag);
#endif
    hconv = DoConv(num, den, hconvRESOLUTION);
    vconv = DoConv(num, den, vconvRESOLUTION);

    NoSignExtend(dvifp, 4);	/* height-plus-depth of tallest page */
    NoSignExtend(dvifp, 4);	/* width of widest page */
    if (NoSignExtend(dvifp, 2) >= STACKSIZE)
	Fatal ("Stack size is too small");
    NoSignExtend(dvifp, 2);	/* this reads the number of pages in */
    /* the DVI file */
#ifdef DEBUG
    if (Debug) fprintf (stderr, "now reading font defs");
#endif
    if (load) GetFontDef ();
}

#ifdef PK

/*-->pk_get_nyb*/
/**********************************************************************/
/*****************************  pk_get_nyb  ******************************/
/**********************************************************************/

int
pk_get_nyb()
{
    if (bit_weight==0) {
	input_byte = getc(pxlfp);
	bit_weight = 8;
    }
    return ((input_byte>>(bit_weight -= 4))&017);
}

/*-->pk_packed_num*/
/**********************************************************************/
/*****************************  pk_packed_num  ******************************/
/**********************************************************************/

int
pk_packed_num()
{
    register int i, j;
    i = pk_get_nyb();
    if (i==0) {
	do {j = pk_get_nyb(); i++;} while (j==0);
	while (i-- > 0) 
	    j = (j<<4) + pk_get_nyb();
	return (j-15+((13-dyn_f)<<4)+dyn_f);
    } else if (i <= dyn_f) return (i);
    else if (i<14) return (((i-dyn_f-1)<<4)+pk_get_nyb()+dyn_f+1);
    else {
	if (i==14) rep_count = pk_packed_num();
	else rep_count = 1;
	return (pk_packed_num());
    }
}
	
/*-->Get_Raster*/
/**********************************************************************/
/*****************************  Get_Raster  ******************************/
/**********************************************************************/

void
get_raster(ptr)
register struct char_entry *ptr;
{
    register int ww, bw, w, j, i, c;
    register char *pr;
	pr = ptr->where.address.pixptr;
	bw = 0;
	for (i=ptr->height; i>0; i--) {
	    w = 0; ww = 0200;
	    for (j=ptr->width; j>0; j--) {
		if ((bw>>=1)==0) { bw = 0200; c = getc(pxlfp);}
		if (c&bw) w |= ww;
		if ((ww>>=1)==0) {*pr++ = w; w = 0; ww = 0200;}
	    }
	    if (ww != 0200) *pr++ = w;
	}
}

/*-->Unpack_Raster*/
/**********************************************************************/
/*****************************  Unpack_Raster  ******************************/
/**********************************************************************/

void
unpack_raster(ptr)
register struct char_entry *ptr;
{
    char row[1000];
    register int ww, w, h_bit, count, r_l, r_p;
    register char *pr, turn_on;
    bit_weight = 0;
    turn_on = ptr->turn_on;
    pr = ptr->where.address.pixptr;
    r_l = ptr->height;
    h_bit = ptr->width;
    rep_count = 0;
    ww = 8; w = 0; r_p = 1;
    while (r_l > 0) {
	count = pk_packed_num();
	while (count > 0) {
	    if ((count<ww) && (count<h_bit)) {
		if (turn_on) w |= gpower[ww] - gpower[ww-count];
		h_bit -= count; ww -= count; count = 0;
	    } else if ((count>=h_bit) && (h_bit<=ww)) {
		if (turn_on) w |= gpower[ww] - gpower[ww-h_bit];
		row[r_p] = w;
		r_l -= (++rep_count);
		while (rep_count-- > 0) {
		    for (w=1; w<=r_p;) *pr++ = row[w++];
		}
		rep_count = 0; r_p = 1; w = 0; ww = 8;
		count -= h_bit; h_bit = ptr->width;
	    } else {
		if (turn_on) w |= gpower[ww];
		row[r_p++] = w;
		w = 0; count -= ww; h_bit -= ww;
		ww = 8;
	    }
	}
	turn_on = ! turn_on;
    }
}
		    

#endif /* PK */

/*-->LoadAChar*/
/**********************************************************************/
/*****************************  LoadAChar  ******************************/
/**********************************************************************/

LoadAChar(c, ptr)
int c;
register struct char_entry *ptr;
{
    char *pr;
    register int nints;

#ifdef BUILTIN
    if (fontptr->builtin) {
      if (fontptr->ncdl == 0 ) {
	register int i;
		    /**********************************************************
		      Font Size is ActualFactor(mag) * s/d * Normal Size
		      Normal Size is	d		(DVI UNITS)
		      * num/den	(10^-7 metres)
		      / 10^-5		(centimetres)
		      / 2.54		(inches)
		      * 300		DOTS
		     *********************************************************/
	GEMIT(OUTFP, "/%s /%s %f TeXPSmakefont def\n", 
	     fontptr->psname,
	     PSname(fontptr->n),
	     ActualFactor(mag) * fontptr->s * num / den
	     / 254000.0 * resolution);
	for (i = FIRSTPXLCHAR; i <= LASTPXLCHAR; i++)
	  fontptr->ch[i].where.isloaded = PERMANENT;
	fontptr->ncdl++;
#ifdef BUDGET
	budget -= fontcost(fontptr);
#endif
      }
    return;
  }
#endif

    if (ptr->where.address.fileOffset == NONEXISTANT) {
      ptr->where.address.pixptr = NULL;
      ptr->where.isloaded = TEMPORARY;
      return;
    }
    
    OpenFontFile();
    fseek(pxlfp, ptr->where.address.fileOffset, 0);
    nints = ((ptr->width + 7) >> 3) * ptr->height;
    if( (pr = (char *)malloc( nints )) == NULL )
      Fatal("Unable to allocate memory for char\n");
    ptr->where.address.pixptr = pr;
    ptr->where.isloaded = TEMPORARY;
#ifndef PK
    fread(pr, 4, nints, pxlfp);
#else
    if ((dyn_f=ptr->dyn_f) == 14) 	/* get bit-raster */
      get_raster(ptr);
    else 
      unpack_raster(ptr);
#endif /* PK */
    
    EmitChar(c, ptr);
}

/*-->SetChar*/
/**********************************************************************/
/*****************************  SetChar  ******************************/
/**********************************************************************/

void
SetChar(c, command, PassNo)
int c, command, PassNo;
{
    register struct char_entry *ptr;  /* temporary char_entry pointer */
    int k;

    ptr = &(fontptr->ch[c]);
    if (ptr->where.isloaded==NOTLOADED) LoadAChar(c, ptr);
    if (PassNo==0) return;

    SetPosn(hh,vv);
    if (fontptr->font_file_id != NO_FILE) {     /* ignore missing fonts */
#ifdef BUDGET
	if (ptr->where.isloaded!=PERMANENT) {	/* and missing characters */
	    if (ptr->where.address.pixptr != NULL) {
		EMITS("save\n");
		EmitBitmap(c, ptr, "@pc restore", FALSE);
	    }
	} else {
#endif
	    if( fontptr != prevfont ) {
		    EMIT(outfp,"%s @sf\n", fontptr->psname);
		    prevfont = fontptr;
		    }
	    EMITN(c); EMITS("c\n");
	    hhh += ptr->dx;
#ifdef BUDGET
	}
#endif
    }

    if (command <= SET4) {
        hh += ptr->dx;
	MoveRight(ptr->tfmw);
    }

#ifdef STATS
    Stnc ++;
    fontptr->ncts ++;
#endif
}


/*-->SetFntNum*/
/**********************************************************************/
/****************************  SetFntNum  *****************************/
/**********************************************************************/

void
SetFntNum(k, Emitting)
int k, Emitting;

/*  this routine is used to specify the font to be used in printing future
    characters */

{
    fontptr = hfontptr;
    while ((fontptr!=NULL) && (fontptr->k!=k))
	fontptr = fontptr->next;
    if (fontptr == NULL)
	Fatal("font %d undefined", k);
/*
    if (Emitting && (fontptr->font_file_id != NO_FILE) )
        EMIT(outfp,"%s @sf\n", fontptr->psname);
*/
}


/*-->SetPosn*/
/**********************************************************************/
/*****************************  SetPosn  ******************************/
/**********************************************************************/

void
SetPosn(x, y)           /* output a positioning command */
int x, y;
{
        int rx,ry;
#ifdef USERELPOS
        if (y == vvv) {  /* use relative movement if just moving horizontally */
            if ( x != hhh ) {
                EMITN(rx=x-hhh);
                EMITS("r ");
                }
            }
        else {
#endif
            EMITN(x);
            EMITN(y);
            EMITS("p ");
#ifdef USERELPOS
            }
#endif
	hhh = x;
	vvv = y;
}


/*-->SetRule*/
/**********************************************************************/
/*****************************  SetRule  ******************************/
/**********************************************************************/

void
SetRule(a, b, Set)
int a, b;
BOOLEAN Set;

{	    /*	 this routine will draw a rule */
    int q;	/* b, rounded to pixels */

    if( a > 0 && b > 0 ) {
        SetPosn(hh,vv);                   /* lower left corner */
        q = RuleRound(b,hconv);      	/* width */
        EMITN( q );       		/* width */
        EMITN(RuleRound(a,vconv));       /* height */
        EMITS("ru\n");
        }
    if (Set) { hh += q; MoveRight(b); }
}


/*-->SetString*/
/**********************************************************************/
/*****************************  SetString  ****************************/
/**********************************************************************/

void
SetString(firstch, PassNo)              /* read and set a consecutive string of chars */
int firstch, PassNo;
{
#define SLEN 256
    char string[SLEN];
    register char *sp;
    register int  c;
    register struct char_entry *ptr;
    int len, h1, hh1;
    int special;	/* 1 if we couldn't load the last character */

    /* read entire string of chars */

    special = 0;
    for(c = firstch, sp = string; c >= SETC_000 && c <= SETC_127 && sp < &string[SLEN]; ) {
      *sp++ = c;
      ptr = &(fontptr->ch[c]);
      if (ptr->where.isloaded==NOTLOADED ) LoadAChar(c, ptr);
      if (PassNo != 0 && ptr->where.isloaded!=PERMANENT)
	{ special = 1; break; }
      c = NoSignExtend(dvifp, 1);
    }
    if (!special) ungetc(c, dvifp);           /* backup one character */
    
    if( PassNo == 0 ) return;
    len = sp - string - special;        	/* NULL's are valid chars, */
    /* so cant use for string termination */
    if( len > 0 &&
       fontptr->font_file_id != NO_FILE ) {     /* ignore missing fonts */
	 sp = string;
	 if( fontptr != prevfont ) {
	   EMIT(outfp,"%s @sf\n", fontptr->psname);
	   prevfont = fontptr;
	 }
	 while (sp < string+len) {
	   if (sp == string+len-1) SetChar (*sp++, SET1, PassNo);
	   else {
	     SetPosn(hh, vv);
	     EMITC('(');
	     h1 = h ;
	     while (sp < string+len) {
	       ptr = &(fontptr->ch[*sp]);
	       h1 += ptr->tfmw;
	       hh += ptr->dx;
	       if( *sp < ' ' || *sp >= 0177 )
		 EMITO(*sp);
	       else if( *sp == '(' || *sp == ')' || *sp == '\\') {
		 EMITC('\\'); 
		 EMITC(*sp); 
	       }
	       else
		 EMITC(*sp);
	       ++sp;
	       hh1 = PixRound (h1, hconv) - hh;
	       if ( abs (hh1) > max_drift) break;
	     }
	     EMITS(") s\n");
	     hhh = hh;
	     MoveRight(h1-h);
	   }
	 }
       }	 
    if (special) SetChar (string[len], SET1, PassNo);

#ifdef STATS
    Stnc += len;
    fontptr->ncts += len;
#endif
}


/*-->SignExtend*/
/**********************************************************************/
/****************************  SignExtend  ****************************/
/**********************************************************************/

int
SignExtend(fp, n)   /* return n byte quantity from file fd */
register FILE *fp;  /* file pointer    */
register int n;     /* number of bytes */

{
    int n1;         /* number of bytes	    */
    register int x; /* number being constructed */

    x = getc(fp);   /* get first (high-order) byte */
    n1 = n--;
    while (n--)  {
	x <<= 8;
	x |= getc(fp);
    }

    /* NOTE: This code assumes that the right-shift is an arithmetic, rather
    than logical, shift which will propagate the sign bit right.   According
    to Kernighan and Ritchie, this is compiler dependent! */

    x<<=32-8*n1;
    x>>=32-8*n1;  /* sign extend */

#ifdef DEBUG
    if (Debug)
    {
	fprintf(stderr,"\tSignExtend(fp,%d)=%X\n",n1,x);
    }
#endif
    return(x);
}


/*-->SkipFontDef*/
/**********************************************************************/
/****************************  SkipFontDef  ***************************/
/**********************************************************************/

void
SkipFontDef(k)
int k;
{
    int a, l;
    char n[STRSIZE];

    NoSignExtend(dvifp, 4);
    NoSignExtend(dvifp, 4);
    NoSignExtend(dvifp, 4);
    a = NoSignExtend(dvifp, 1);
    l = NoSignExtend(dvifp, 1);
    GetBytes(dvifp, n, a+l);
}


/*-->Warning*/
/**********************************************************************/
/*****************************  Warning  ******************************/
/**********************************************************************/

void
Warning(fmt, a, b, c)  /* issue a warning */
char *fmt;	/* format   */
char *a, *b, *c;	/* arguments */
{
    if (G_logging == 0)
    {
        if (G_logfile)
	        G_logfp=fopen(G_Logname,"w+");
        else {
                G_logfp=stderr;
                if( G_nowarn ) return;
                }
	G_logging = 1;
	if (G_logfp == NULL) G_logging = -1;
    }

    G_errenc = TRUE;
    if (G_logging == 1)
    {
	fprintf(G_logfp, fmt, a, b, c);
	fprintf(G_logfp,"\n");
    }
}


