// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: hp2686.C,v 1.20 91/02/22 15:55:21 hmgr Exp $";

// HP2686 LaserJet+ and LaserJetII device-specific functions.
// The various Reference Manuals should be consulted for more
// information on the various escape-sequences.
//
// by Parag Patel

#include "defs.h"
#include "hp2686.h"


static int hoffset = HP2686_HOFFSET;
static int voffset = HP2686_VOFFSET;


// put a 16-bit word on the output as two bytes
// 
inline void put16(long n)
{
    putchar((char)((n >> 8) & 0xFF));
    putchar((char)(n & 0xFF));
}

static void initlj()
{
    if (landscape)
    {
	fputs("\033&l1O", stdout);
	hoffset -= 60;
	voffset -= 10;
    }
    else
	fputs("\033&l0O", stdout);
    fputs("\033&l0E", stdout);
}

HP2686::HP2686(int maxf, int maxc, int maxpg, int maxld, int res,
		int ho, int vo, int fhmn, int fhmx, int fvmn, int fvmx,
		int fwd, int fhg, char *deffp) :
    Device(maxf, maxc, maxpg, maxld, res, ho, vo,
	    fhmn, fhmx, fvmn, fvmx, fwd, fhg, deffp)
{
    initlj();
}

// initialize the LaserJet
// 
HP2686::HP2686() :
    Device(257, 256, 16, 32, 300, hoffset, voffset,
	    -128, 127, -128, 127, 128, 128, HP2686_FPATH)
{
    initlj();
}

// clean up after dumping a DVI file
// 
HP2686::~HP2686()
{
    fputs("\033&l0O", stdout);
}

// convert a character number to something that the device can deal with
// for a LaserJet
//
const char *HP2686::char2dev(int ch, int &len)
{
    static char buf[20];
    if ((ch >= 32 && ch <= 127) || ch >= 160)
	sprintf(buf, "%c", ch);
    else
	sprintf(buf, "\033&p1X%c", ch);
    len = strlen(buf);
    if (ch == 0)
	len++;
    return buf;
}

// convert a scaled point to a device pixel coordinate
//
long HP2686::sp2dev(double sp)
{
    return long(sp / 72.27 * 300.0 / double(1 << 16L) + 0.5);
}

// convert a device pixel coordinate to a scaled point
//
double HP2686::dev2sp(long pix)
{
    return double(pix) / 300.0 * 72.27 * double(1 << 16L);
}


// download a big character to the LaserJet in graphics mode
// 
void HP2686::bigchar(font &, fontchar &g, int)
{
    long width = g.maxm - g.minm + 1;
    long height = g.maxn - g.minn + 1;
    int res = 1;		// 1==300dpi 2==150dpi

    makecurrent();
    push();			// save current location

    // move (relative) up - printf() may not handle the sign properly
    // with the "%+ld" specification, so we do it ourselves
    long rel = -g.maxn;
    if (rel < 0)
	printf("\033*p%ldY", rel);
    else
	printf("\033*p+%ldY", rel);

    // move (relative) right
    rel = landscape ? g.maxm : g.minm;
    if (rel < 0)
	printf("\033*p%ldX", rel);
    else
	printf("\033*p+%ldX", rel);

    if (width > 300 || height > 300)
    {
	// use 150 dots per inch for REALLY big characters
	res = 2;
	fputs("\033*t150R", stdout);
    }
    else
	fputs("\033*t300R", stdout);
    fputs("\033*r1A", stdout);	// start graphics mode

    if (landscape)
	for (register long j = g.maxm - g.minm; j >= 0; j -= res)
	{
	    int b = 0;
	    int bp = 0x80;
	    register long count = (height + 7) >> (2 + res);

	    printf("\033*b%ldW", count);

	    for (register long i = g.maxn - g.minn; i >= 0; i -= res)
	    {
		if (bp == 0)
		{
		    putchar(b);
		    bp = 0x80;
		    b = 0;
		    count--;
		}
		if (res == 1)	// 300 dpi
		{
		    if (fontbits[i]->isin(j))
			b |= bp;
		}
		else		// 150 dpi
		{
		    // use the "or" of the 4 pixels at this location
		    if (fontbits[i]->isin(j)
			    || fontbits[i - 1]->isin(j)
			    || fontbits[i]->isin(j + 1)
			    || fontbits[i - 1]->isin(j + 1))
			b |= bp;
		}
		bp >>= 1;
	    }

	    putchar(b);
	    if (count != 1 && count != 0)
		quit("Code for downloading bigchars is hosed: count=%ld",
			count);
	}
    else
	// we use "res" as an incrementer for 150 or 300 dpi
	for (register long i = g.maxn - g.minn; i >= 0; i -= res)
	{
	    int b = 0;
	    int bp = 0x80;
	    register long count = (width + 7) >> (2 + res);

	    printf("\033*b%ldW", count);

	    for (register long j = 0; j <= g.maxm - g.minm; j += res)
	    {
		if (bp == 0)
		{
		    putchar(b);
		    bp = 0x80;
		    b = 0;
		    count--;
		}
		if (res == 1)// 300 dpi
		{
		    if (fontbits[i]->isin(j))
			b |= bp;
		}
		else		// 150 dpi
		{
		    // use the "or" of the 4 pixels at this location
		    if (fontbits[i]->isin(j)
			    || fontbits[i - 1]->isin(j)
			    || fontbits[i]->isin(j + 1)
			    || fontbits[i - 1]->isin(j + 1))
			b |= bp;
		}
		bp >>= 1;
	    }

	    putchar(b);
	    if (count != 1 && count != 0)
		quit("Code for downloading bigchars is hosed: count=%ld",
			count);
	}

    fputs("\033*rB", stdout);	// end graphics mode
    pop();			// restore saved location
}


// download a character for a particular font
// 
void HP2686::downchar(font &f, fontchar &g, int ch)
{
    // describe the character and its font to the LaserJet
    printf("\033*c%ldd%dE", f.num, ch);
    downljchar(f, g, ch);
}

// this sets up this character in the LaserJet memory so that
// it can be typeset later
// - assumes that the font has already been downloaded to the printer
// 
void HP2686::downljchar(font &, fontchar &g, int)
{
    long width = g.maxm - g.minm + 1;
    long height = g.maxn - g.minn + 1;

    // number of bytes for a bit-map of the character
    long count;
    if (landscape)
    {
	count = width * ((height + 7) >> 3);
	if (count < 1)
	    count = 1;
    }
    else
	count = height * ((width + 7) >> 3);

    printf("\033(s%ldW", count + 16);
    putchar(4);			// these 4 bytes are magic -
    putchar(0);			// they must be here for some
    putchar(14);		// unknown reason...
    putchar(1);

    if (landscape)
    {
	putchar(1);		// landscape mode
	putchar(0);		// <magic>
	put16(-g.maxn);		// # pixels above the baseline
	put16(g.maxm);		// # pixels to the right of origin
	put16(height);		// width of char in pixels
	put16(width);		// height in pixels
    }
    else
    {
	put16(0);		// portrait mode | magic
	put16(g.minm);		// # pixels to the left of origin
	put16(g.maxn);		// # pixels above the baseline
	put16(width);		// width of char in pixels
	put16(height);		// height in pixels
    }
    put16(g.dx >> 14);		// delta-x in pixels * 4

    // download the bitmap for this character one byte at a time - we
    // also decrement the "count" var to make sure that we send down
    // exactly the correct number of bytes to the LaserJet
    // (OK OK - there are more effecient ways to do this, but this works
    // and is simple to code.  It'll do for now.)

    register long n;
    register long m;
    if (landscape)
	for (m = g.maxm - g.minm; m >= 0; m--)
	{
	    // we will build the byte "b" to download - "bp" is the bit
	    // pointer used to set the next bit in the current byte - it
	    // is shifted by one each time around the inner loop
	    int b = 0;
	    int bp = 0x80;

	    for (n = g.maxn - g.minn; n >= 0; n--)
	    {
		if (bp == 0)
		{
		    // we have put the last bit that this byte can hold
		    // into "b" - download it, then reset the bit pointer
		    putchar(b);
		    bp = 0x80;
		    b = 0;
		    count--;
		}

		// set this bit in the byte if the current bit is set in
		// the bitmap - then shift bp to point to the next bit
		if (fontbits[n]->isin(m))
		    b |= bp;
		bp >>= 1;
	    }

	    // put the remaining byte out - note that the last byte will
	    // always be downloaded here and not in the previous loop
	    putchar(b);
	    count--;
	}
    else
	for (n = g.maxn - g.minn; n >= 0; n--)
	{
	    // we will build the byte "b" to download - "bp" is the bit
	    // pointer used to set the next bit in the current byte - it
	    // is shifted by one each time around the inner loop
	    int b = 0;
	    int bp = 0x80;

	    for (m = 0; m <= g.maxm - g.minm; m++)
	    {
		if (bp == 0)
		{
		    // we have put the last bit that this byte can hold
		    // into "b" - download it, then reset the bit pointer
		    putchar(b);
		    bp = 0x80;
		    b = 0;
		    count--;
		}

		// set this bit in the byte if the current bit is set in
		// the bitmap - then shift bp to point to the next bit
		if (fontbits[n]->isin(m))
		    b |= bp;
		bp >>= 1;
	    }

	    // put the remaining byte out - note that the last byte will
	    // always be downloaded here and not in the previous loop
	    putchar(b);
	    count--;
	}

    // see if we have a bug in the above code
    if (count != 0)
	quit("Code for downloading chars is hosed: count=%ld", count);
}


// flush out (print) the current page in the LaserJet - get ready to
// print a new page
// 
void HP2686::newpage(boolean odd, boolean first)
{
    if (first)
	return;
    odd = FALSE;
    putchar('\f');
}


// move the LaserJet cursor to the specified (h,v) spot on the paper
// - add the appropriate "fuhj" offsets to the coordinates
// 
void HP2686::movehv(long h, long v)
{
    printf("\033*p%ldx%ldY", h + hoffset, v + voffset);
}


// just move along the horizontal to the specified pixel
// 
void HP2686::moveh(long h)
{
    printf("\033*p%ldX", h + hoffset);
}


// vertical move
// 
void HP2686::movev(long v)
{
    printf("\033*p%ldY", v + voffset);
}


// push the current spot - fortunately the LaserJet already does this
// or we would save the current location in our own stack
// 
void HP2686::push()
{
    fputs("\033&f0S", stdout);
}


// pop the current location and restore the last one
// 
void HP2686::pop()
{
    fputs("\033&f1S", stdout);
}


// delete the specified font from the LaserJet's memory
// 
void HP2686::delfont(font &f)
{
    printf("\033*c%ldd2F", f.num);
}


// download this font to the LaserJet's memory - this just defines
// the font but doesn't actually download any characters into it yet
// 
void HP2686::newfont(font &f)
{
    long width = f.maxm - f.minm + 1;
    if (width < 1)
	width = 1;
    long height = f.maxn - f.minn + 1;
    if (height < 1)
	height = 1;

    printf("\033*c%ldD\033)s26W", f.num);    // font number
    put16(26);			// length of the this+rest in bytes
    put16(1);			// 0 <magic>  |  font-type
    put16(0);			// <magic>
    put16(f.maxn);		// baseline position
    put16(width);		// cell width
    put16(height);		// cell height
    putchar(landscape ? 1 : 0);	// orientation
    putchar(1);			// 		fixed/proportional
    putchar(1);			// symbol set
    putchar(21);		// 		symbol set II
    put16(width << 2);		// pitch (default HMI)
    put16(height << 2);		// height
    put16(0);			// 0 <magic>
    put16(0);			// 0 <magic> | style
    put16(0);			// stroke width | typeface
}


// make this font the current font in the LaserJet - we are about to
// typeset (print) some characters from it
// 
void HP2686::usefont(font &f)
{
    printf("\033(%ldX", f.num);
}


// typeset (print) a rule (box) of the specified length and width at
// the current cursor location
// 
void HP2686::rule(double A, double B)
{
    long aa = sp2dev(A);
    long bb = sp2dev(B);

    debug(5, "rule: a=%f b=%f  H=%f V=%f  aa=%ld bb=%ld", A, B, H, V, aa, bb);

    // now typeset the rule - again the LaserJet makes this easy with
    // the fill area command - unfortunately the origin must be at the
    // top-left corner and not the bottom-left like a normal character
    movedown(-A);
    makecurrent();
    printf("\033*c%lda%ldb0P", bb <= 0 ? 1 : bb, aa <= 0 ? 1 : aa);
    movedown(A);
}
