;;;
;;; hebrew.el -- Hebrew editing mode.
;;; For more information, see documentation of `hebrew-mode'.
;;;
;;; Copyright (c) Yossi Friedman, April 1990
;;;

(defvar running-epoch nil "*Non-nil means Epoch is running.")

(defvar hebrew-mode-hook () "*Called when entering hebrew-mode")

(defvar hebrew-mode nil "*Non nil when Hebrew mode is turned on.")
(make-variable-buffer-local 'hebrew-mode)

(or (assoc 'hebrew-mode minor-mode-alist)
    (setq minor-mode-alist
	  (cons '(hebrew-mode " Hebrew") minor-mode-alist)))

(defun hebrew-mode ()
"\
Toggles hebrew-mode on/off.  Hebrew-mode is a minor mode which allows
reversing the editor display and changing the keyboard in order to
enter and edit text in a semitic language, such as Hebrew.

By default, hebrew-mode stays in \"normal\" state.  The \"reversed\"
state is activated with \"\\[hebrew-toggle-reversed]\".  (In Epoch, this
is also bound to \"F3\".)  In the \"reversed\" state, hebrew-mode emulates
a Hebrew keyboard both during direct insertion and during searches.

Hebrew-mode calls hebrew-mode-hook upon entering."


  (interactive)
  (if (null hebrew-mode)
      (let (char)

	; signal hebrew minor mode in the mode-line
	(setq hebrew-mode t)

	; activate the hebrew insertion function
	(setq char ? )
	(while (<= char ?~)
	  (local-set-key (format "%c" char) 'hebrew-insert)
	  (setq char (1+ char)))

	; display-literal, display-reversed
	(setq display-literal t)
	(local-set-key "\C-cR" 'hebrew-toggle-reversed)

	; syntax table stuff
	(setq char ?\340)	; aleph
	(while (<= char ?\372)	; taf
	  (modify-syntax-entry char "w")
	  (setq char (1+ char)))

	; search stuff
	(local-set-key "\C-s" 'hebrew-ctrl-s)
	(local-set-key "\C-r" 'hebrew-ctrl-r)

	; hooks
	(run-hooks 'hebrew-mode-hook))

    (let (char)

      ; turn off hebrew minor mode in the mode-line
      (setq hebrew-mode nil)

      ; deactivate the hebrew insertion function
      (setq char ? )
      (while (<= char ?~)
	(local-unset-key (format "%c" char))
	(setq char (1+ char)))

      ; deactivate display-literal and display-reversed
      (setq display-literal nil)
      (local-unset-key "\C-cR")

      ; search stuff
      (local-unset-key "\C-s")
      (local-unset-key "\C-r")

      ; restore syntax-table
      (setq char ?\340)		; aleph
      (while (<= char ?\372)	; taf
	(modify-syntax-entry char " ")
	(setq char (1+ char))))))

(if running-epoch
    (epoch::rebind-key "F3" 0 "\C-cR"))
(defun hebrew-toggle-reversed ()
  "Toggle whether or not the display is laterally reversed."
  (interactive)
  (setq display-reversed (null display-reversed))
  (if running-epoch
      (epoch::redisplay-screen)
    (x-smart-redisplay)))

(defun hebrew-insert (&optional arg)
  "\
If display-reversed is nil, behaves like self-insert-command.
If display-reversed is t, simulates a Hebrew typewriter keyboard."

  (interactive)
  (or (numberp arg) (setq arg 1))
  (let ((char (if display-reversed
		  (aref hebrew-keyboard last-command-char)
		(format "%c" last-command-char))))
    (while (> arg 0)
      (insert char)
      (setq arg (1- arg)))))

(defun hebrew-ctrl-s () "The Hebrew-mode version of \C-s"
  (interactive)
  (let ((old-read-char (symbol-function 'read-char))
	(old-text-char-description (symbol-function 'text-char-description))
	(search-exit-option nil)
	res)
    (fset 'read-char (symbol-function 'hebrew-read-char))
    (fset 'text-char-description
	  (symbol-function 'hebrew-text-char-description))
    (setq res (funcall (global-key-binding "\C-s")))
    (fset 'read-char old-read-char)
    (fset 'text-char-description old-text-char-description)
    res))

(defun hebrew-ctrl-r () "The Hebrew-mode version of \C-r"
  (interactive)
  (let ((old-read-char (symbol-function 'read-char))
	(old-text-char-description (symbol-function 'text-char-description))
	(search-exit-option nil)
	res)
    (fset 'read-char (symbol-function 'hebrew-read-char))
    (fset 'text-char-description
	  (symbol-function 'hebrew-text-char-description))
    (setq res (funcall (global-key-binding "\C-r")))
    (fset 'read-char old-read-char)
    (fset 'text-char-description old-text-char-description)
    res))

(defun hebrew-read-char ()
  "The hebrew-mode version of read-char."
  (let ((char (funcall old-read-char)))
    (if (and display-reversed (>= char ? ) (<= char ?~))
	(string-to-char (aref hebrew-keyboard char))
      char)))

(defun hebrew-text-char-description (char)
  "The hebrew-mode version of text-char-description."
  (if display-reversed
      (char-to-string char)
    (funcall old-text-char-description char)))

; for the search function to work, we can't use a byte-compiled version of
; isearch.el, so load it back:
(load-library "isearch.el")

(setq hebrew-keyboard
      [
       ""     ""     ""     ""     ""     ""     ""     ""
       ""     ""     ""     ""     ""     ""     ""     ""
       ""     ""     ""     ""     ""     ""     ""     ""
       ""     ""     ""     ""     ""     ""     ""     ""
       " "    "!"    "\""   "#"    "$"    "%"    "&"    "'"
       "("    ")"    "*"    "+"    "\372" "-"    "\365" "."
       "0"    "1"    "2"    "3"    "4"    "5"    "6"    "7"
       "8"    "9"    ":"    "\363" "<"    "="    ">"    "?"
       "@"    "A"    "B"    "C"    "D"    "E"    "F"    "G"
       "H"    "I"    "J"    "K"    "L"    "M"    "N"    "O"
       "P"    "Q"    "R"    "S"    "T"    "U"    "V"    "W"
       "X"    "Y"    "Z"    "["    "\\"   "]"    "^"    "_"
       ";"    "\371" "\360" "\341" "\342" "\367" "\353" "\362"
       "\351" "\357" "\347" "\354" "\352" "\366" "\356" "\355"
       "\364" "\\"   "\370" "\343" "\340" "\345" "\344" ","
       "\361" "\350" "\346" "{"    "|"    "}"    "~"    ""])
