/*
** Datei: dviapollo.c
** Autor: Ingo Eichenseher
*/      

#include <apollo/base.h>
#include <apollo/gpr.h>
#include <apollo/kbd.h>
#include <apollo/tone.h>  
#include <apollo/error.h>

#define SETSIZE (short)256      

#include <ctype.h>       
#include <stdio.h>
#include <stdarg.h>
#include "dvi.h"    
#include "dvihdcp.h"
#include "dviframe.h"

#define XXSTEP 100
#define YYSTEP 100
#define XSTEP 8
#define YSTEP 8


static gpr_$bitmap_desc_t       memory_bitmap;
static gpr_$offset_t            memory_size;
static short int            memory_groups;
static gpr_$attribute_desc_t        memory_attrib;
static gpr_$bmf_group_header_array_t    memory_g_headers;
static gpr_$color_vector_t      color_map;
static status_$t            status;

void check(char *messagex)
{
    if (status.all)
    {   
	error_$print (status);
	fprintf(stderr,"Error occurred while %s.\n", messagex);
    }    
}

static void create_array(void)
{
    gpr_$allocate_attribute_block(&memory_attrib,&status);
    check("Allocating");
    memory_size.x_size = frame_width*8;
    memory_size.y_size = frame_height; 
    memory_groups = 1;
    memory_g_headers[0].pixel_size = 1;
    memory_g_headers[0].n_sects = 1;
    memory_g_headers[0].allocated_size = 0;
    memory_g_headers[0].bytes_per_line = 0;
    memory_g_headers[0].bytes_per_sect = 0;
    memory_g_headers[0].storage_offset = (char *)frame_ptr(0);

    gpr_$make_bitmap_from_array(memory_size,memory_attrib,memory_groups,
	memory_g_headers,&memory_bitmap,&status);     
    check("Making bitmap");
}

static void display_array(int x, int y)
{
    gpr_$window_t   src_window;
    gpr_$position_t dst_offset;

    src_window.window_size.x_size = frame_width*8;
    src_window.window_size.y_size = frame_height;
    src_window.window_base.x_coord = x;
    src_window.window_base.y_coord = y;
    dst_offset.x_coord = 0;
    dst_offset.y_coord = 0;                                  

    gpr_$pixel_blt(memory_bitmap,src_window,dst_offset,&status); 
    check("BLTing");
}                                                 

static int xoff = 0, yoff = 0, xsize, ysize;

static void check_position(void)
{
    gpr_$disp_char_t    display_characteristics;
    long            pad_id = 1;
    static short int    disp_len = sizeof(gpr_$disp_char_t);
    short int       disp_len_returned;

    gpr_$inq_display_characteristics(gpr_$pad_id,pad_id,disp_len,
	&display_characteristics,&disp_len_returned,&status); 
    check("Inquiring");

    if (ysize-yoff < display_characteristics.y_window_size) 
	yoff = ysize - display_characteristics.y_window_size;
    if (xsize-xoff < display_characteristics.x_window_size) 
	xoff = xsize - display_characteristics.x_window_size;
    if (yoff<0) yoff = 0;
    if (xoff<0) xoff = 0;   
}

void refresh(boolean *unobscured, boolean *pos_change)
{
    check_position();
    gpr_$acquire_display(&status);
    display_array(xoff,yoff);
    gpr_$release_display(&status);
}         

void refresh_obscured(void)
{         
    static gpr_$window_t vis_list[50];    
    short int slots_total, i;

    check_position();
    gpr_$acquire_display(&status);      
    check("Acquiring display");
    gpr_$inq_vis_list(50,&slots_total,vis_list,&status);
    check("Inquiring visible list");
    for (i=0; i<slots_total; i++) 
    {
	gpr_$set_clip_window(vis_list[i],&status);
	display_array(xoff,yoff);
    }
    gpr_$release_display(&status);
    check("Releasing display");
}

int screen(void)
{     
    gpr_$pixel_format_t pix_format;
    gpr_$proj_format_t  proj_format;
    gpr_$video_format_t video_format;
    gpr_$bitmap_desc_t  display_bitmap;
    gpr_$offset_t       display_bitmap_size;

    gpr_$event_t        my_events[2];
    gpr_$keyset_t       key_set;

    gpr_$event_t        event_type;
    unsigned char       event_data;

    gpr_$position_t     position;

    boolean         stop; 
    boolean         end = false; 
    boolean         quit = false;     
    boolean         unob, pos_ch;


    xoff = 0; 
    yoff = 0;

    xsize = display_bitmap_size.x_size = frame_width*8; 
    ysize = display_bitmap_size.y_size = frame_height;

    pix_format.pixel_mode = gpr_$pixel_pseudocolor;
    pix_format.image_depth = 1;
    pix_format.length = 2;

    proj_format.proj_mode = gpr_$proj_mode_argb;
    proj_format.proj_buffer = 1;
    proj_format.length = 2;

    video_format.length = 1;
    video_format.video_buffer = 1;

    gpr_$initialize(gpr_$pad_id, 1, 0,
	&display_bitmap_size,&pix_format,&proj_format,&video_format,
	&display_bitmap, &status);    
    check("Initializing");

    create_array(); 

    gpr_$set_refresh_entry(refresh,refresh_obscured,&status); 
    check("Setting refresh entries");
    gpr_$set_obscured_opt(gpr_$ok_if_obs,&status); 
    check("Setting obscured options"); 

    while(!end)
    {
	refresh(&unob,&pos_ch);

	gpr_$set_cursor_active(true,&status);
	my_events[0] = gpr_$locator_stop;
	gpr_$enable_input(my_events[0],key_set,&status);
	lib_$init_set(key_set,SETSIZE);
	lib_$add_to_set(key_set,SETSIZE,'q');
	lib_$add_to_set(key_set,SETSIZE,'n');
	lib_$add_to_set(key_set,SETSIZE,'t');
	lib_$add_to_set(key_set,SETSIZE,KBD_$DOWN_BOX_ARROW2);
	lib_$add_to_set(key_set,SETSIZE,KBD_$R_BOX_ARROW);
	lib_$add_to_set(key_set,SETSIZE,KBD_$L_BOX_ARROW);
	lib_$add_to_set(key_set,SETSIZE,KBD_$UP_BOX_ARROW2);
	lib_$add_to_set(key_set,SETSIZE,KBD_$L8S);
	lib_$add_to_set(key_set,SETSIZE,KBD_$LAS);
	lib_$add_to_set(key_set,SETSIZE,KBD_$LCS);
	lib_$add_to_set(key_set,SETSIZE,KBD_$LES);
	my_events[1] = gpr_$keystroke;
	gpr_$enable_input(my_events[1],key_set,&status);

	stop = false;
	while(!stop)
	{
	    gpr_$event_wait(&event_type,&event_data,&position,&status);
	    switch(event_type)
	    {     
		case gpr_$keystroke    : 
		    switch(event_data)
		    {             
		    case 'n' : stop = end = true; break;
		    case 'q' : stop = end = quit = true; break;
		    case 't' : xoff = yoff = 0; stop = true; break; 
		    case KBD_$L_BOX_ARROW : xoff -= XXSTEP; stop = true; break;
		    case KBD_$R_BOX_ARROW : xoff += XXSTEP; stop = true; break;
		    case KBD_$DOWN_BOX_ARROW2: yoff += YYSTEP; stop=true; break;
		    case KBD_$UP_BOX_ARROW2: yoff -= YYSTEP; stop=true; break;
		    case KBD_$L8S : yoff -= YSTEP; stop = true; break;
		    case KBD_$LES : yoff += YSTEP; stop = true; break;
		    case KBD_$LAS : xoff -= XSTEP; stop = true; break;
		    case KBD_$LCS : xoff += XSTEP; stop = true; break;
		    }
		    break;
		default : break;
	    }
	}        
    }
    gpr_$deallocate_attribute_block(memory_attrib,&status);    
    check("Deallocating attribute block");
    gpr_$deallocate_bitmap(memory_bitmap,&status);   
    check("Deallocating bitmap");
    gpr_$terminate(false,&status);
    check("Terminating"); 
    return quit;
}


/***********************************************************************
 Compare strings (ignoring case), and return:
    s1>s2:  >0
    s1==s2:  0
    s1<s2:  <0
***********************************************************************/

/* toupper() is supposed to work for all letters, but PCC-20 does it
incorrectly if the argument is not already lowercase; this definition
fixes that. */

#define UC(c) (islower(c) ? toupper(c) : c)

int stricmp(char *s1, char *s2)
{
    while ((*s1) && (UC(*s1) == UC(*s2)))
    {
    s1++;
    s2++;
    }
    return((int)(UC(*s1) - UC(*s2)));
}
#undef UC

int stop_key(void)
{
    return 0;
}

void install(void)
{
}

void destall(void)
{
}

int getch(void)
{
    fflush(stdout);
	return getchar();
}

void prbyte(int c)
{
}

